#include "settingsmanager.h"
#include <QRegularExpression>

SettingsManager::SettingsManager(QObject *parent)
    : QObject(parent)
    , m_settings("JordiEducador", "IoT02Controller")
{
    loadDefaults();
    load();
}

void SettingsManager::loadDefaults()
{
    // Valors per defecte MQTT
    m_mqttMac = "123456789A11";
    m_mqttBroker = "wss://broker.emqx.io:8084/mqtt";
    m_mqttUser = "";
    m_mqttPassword = "";
    
    // Valors per defecte LDR
    m_ldrRandom = false;
    m_ldrMin = 100;
    m_ldrMax = 3000;
    
    // Valors per defecte BME280
    m_bmeRandom = false;
    m_tempMin = 15;
    m_tempMax = 30;
    m_rhMin = 30;
    m_rhMax = 70;
    m_pressureMin = 990;
    m_pressureMax = 1020;
}

void SettingsManager::load()
{
    m_mqttMac = m_settings.value("mqtt/mac", m_mqttMac).toString();
    m_mqttBroker = m_settings.value("mqtt/broker", m_mqttBroker).toString();
    m_mqttUser = m_settings.value("mqtt/user", m_mqttUser).toString();
    m_mqttPassword = m_settings.value("mqtt/password", m_mqttPassword).toString();
    
    m_ldrRandom = m_settings.value("sensors/ldr_random", m_ldrRandom).toBool();
    m_ldrMin = m_settings.value("sensors/ldr_min", m_ldrMin).toInt();
    m_ldrMax = m_settings.value("sensors/ldr_max", m_ldrMax).toInt();
    
    m_bmeRandom = m_settings.value("sensors/bme_random", m_bmeRandom).toBool();
    m_tempMin = m_settings.value("sensors/temp_min", m_tempMin).toInt();
    m_tempMax = m_settings.value("sensors/temp_max", m_tempMax).toInt();
    m_rhMin = m_settings.value("sensors/rh_min", m_rhMin).toInt();
    m_rhMax = m_settings.value("sensors/rh_max", m_rhMax).toInt();
    m_pressureMin = m_settings.value("sensors/pressure_min", m_pressureMin).toInt();
    m_pressureMax = m_settings.value("sensors/pressure_max", m_pressureMax).toInt();
    
    // Emetre signals per actualitzar QML
    emit mqttMacChanged();
    emit mqttBrokerChanged();
    emit mqttUserChanged();
    emit mqttPasswordChanged();
    
    emit ldrRandomChanged();
    emit ldrMinChanged();
    emit ldrMaxChanged();
    
    emit bmeRandomChanged();
    emit tempMinChanged();
    emit tempMaxChanged();
    emit rhMinChanged();
    emit rhMaxChanged();
    emit pressureMinChanged();
    emit pressureMaxChanged();
}

void SettingsManager::save()
{
    m_settings.setValue("mqtt/mac", m_mqttMac);
    m_settings.setValue("mqtt/broker", m_mqttBroker);
    m_settings.setValue("mqtt/user", m_mqttUser);
    m_settings.setValue("mqtt/password", m_mqttPassword);
    
    m_settings.setValue("sensors/ldr_random", m_ldrRandom);
    m_settings.setValue("sensors/ldr_min", m_ldrMin);
    m_settings.setValue("sensors/ldr_max", m_ldrMax);
    
    m_settings.setValue("sensors/bme_random", m_bmeRandom);
    m_settings.setValue("sensors/temp_min", m_tempMin);
    m_settings.setValue("sensors/temp_max", m_tempMax);
    m_settings.setValue("sensors/rh_min", m_rhMin);
    m_settings.setValue("sensors/rh_max", m_rhMax);
    m_settings.setValue("sensors/pressure_min", m_pressureMin);
    m_settings.setValue("sensors/pressure_max", m_pressureMax);
    
    m_settings.setValue("app/first_run", false);
    m_settings.sync();
}

void SettingsManager::cancel()
{
    load();
}

bool SettingsManager::isFirstRun() const
{
    return m_settings.value("app/first_run", true).toBool();
}

QString SettingsManager::normalizeMac(const QString &mac)
{
    // Elimina ":" i converteix a majúscules
    QString normalized = mac.toUpper();
    normalized.remove(":");
    return normalized;
}

bool SettingsManager::validateMac(const QString &mac)
{
    // Accepta amb o sense ":"
    QRegularExpression macRegex("^([0-9A-Fa-f]{2}[:]?){5}[0-9A-Fa-f]{2}$");
    return macRegex.match(mac).hasMatch();
}

// Setters MQTT
void SettingsManager::setMqttMac(const QString &mac)
{
    QString normalized = normalizeMac(mac);
    if (m_mqttMac != normalized && validateMac(mac)) {
        m_mqttMac = normalized;
        emit mqttMacChanged();
    }
}

void SettingsManager::setMqttBroker(const QString &broker)
{
    if (m_mqttBroker != broker) {
        m_mqttBroker = broker;
        emit mqttBrokerChanged();
    }
}

void SettingsManager::setMqttUser(const QString &user)
{
    if (m_mqttUser != user) {
        m_mqttUser = user;
        emit mqttUserChanged();
    }
}

void SettingsManager::setMqttPassword(const QString &password)
{
    if (m_mqttPassword != password) {
        m_mqttPassword = password;
        emit mqttPasswordChanged();
    }
}

// Setters LDR
void SettingsManager::setLdrRandom(bool random)
{
    if (m_ldrRandom != random) {
        m_ldrRandom = random;
        emit ldrRandomChanged();
    }
}

void SettingsManager::setLdrMin(int min)
{
    if (min >= 0 && min <= m_ldrMax && m_ldrMin != min) {
        m_ldrMin = min;
        emit ldrMinChanged();
    }
}

void SettingsManager::setLdrMax(int max)
{
    if (max <= 4095 && max >= m_ldrMin && m_ldrMax != max) {
        m_ldrMax = max;
        emit ldrMaxChanged();
    }
}

// Setters BME280
void SettingsManager::setBmeRandom(bool random)
{
    if (m_bmeRandom != random) {
        m_bmeRandom = random;
        emit bmeRandomChanged();
    }
}

void SettingsManager::setTempMin(int min)
{
    if (min >= -40 && min <= m_tempMax && m_tempMin != min) {
        m_tempMin = min;
        emit tempMinChanged();
    }
}

void SettingsManager::setTempMax(int max)
{
    if (max <= 85 && max >= m_tempMin && m_tempMax != max) {
        m_tempMax = max;
        emit tempMaxChanged();
    }
}

void SettingsManager::setRhMin(int min)
{
    if (min >= 0 && min <= m_rhMax && m_rhMin != min) {
        m_rhMin = min;
        emit rhMinChanged();
    }
}

void SettingsManager::setRhMax(int max)
{
    if (max <= 100 && max >= m_rhMin && m_rhMax != max) {
        m_rhMax = max;
        emit rhMaxChanged();
    }
}

void SettingsManager::setPressureMin(int min)
{
    if (min >= 870 && min <= m_pressureMax && m_pressureMin != min) {
        m_pressureMin = min;
        emit pressureMinChanged();
    }
}

void SettingsManager::setPressureMax(int max)
{
    if (max <= 1085 && max >= m_pressureMin && m_pressureMax != max) {
        m_pressureMax = max;
        emit pressureMaxChanged();
    }
}
