#ifndef MQTTCONTROLLER_H
#define MQTTCONTROLLER_H

#include <QObject>
#include "mqttbasic.h"
#include "settingsmanager.h"

class MqttController : public QObject
{
    Q_OBJECT
    
    // Estats de connexió
    Q_PROPERTY(ConnectionState connectionState READ connectionState NOTIFY connectionStateChanged)
    Q_PROPERTY(QString connectionText READ connectionText NOTIFY connectionTextChanged)
    Q_PROPERTY(QString connectionColor READ connectionColor NOTIFY connectionColorChanged)
    
    // Estats dels LEDs (controlats per MQTT)
    Q_PROPERTY(bool ledWState READ ledWState NOTIFY ledWStateChanged)
    Q_PROPERTY(bool ledRState READ ledRState NOTIFY ledRStateChanged)
    Q_PROPERTY(bool ledYState READ ledYState NOTIFY ledYStateChanged)
    Q_PROPERTY(bool ledGState READ ledGState NOTIFY ledGStateChanged)
    
    // Estat de l'OLED
    Q_PROPERTY(QString oledText READ oledText NOTIFY oledTextChanged)
    Q_PROPERTY(int oledTextSize READ oledTextSize NOTIFY oledTextSizeChanged)

    // Estat del sensor LDR
    Q_PROPERTY(int ldrValue READ ldrValue NOTIFY ldrValueChanged)

    // Estat del sensor BME280
    Q_PROPERTY(float tempValue READ tempValue NOTIFY tempValueChanged)
    Q_PROPERTY(int rhValue READ rhValue NOTIFY rhValueChanged)
    Q_PROPERTY(int pressureValue READ pressureValue NOTIFY pressureValueChanged)

public:
    enum ConnectionState {
        Disconnected = 0,
        Connecting = 1,
        Connected = 2
    };
    Q_ENUM(ConnectionState)
    
    explicit MqttController(SettingsManager *settings, QObject *parent = nullptr);
    ~MqttController();
    
    // Getters
    ConnectionState connectionState() const { return m_connectionState; }
    QString connectionText() const;
    QString connectionColor() const;
    
    bool ledWState() const { return m_ledWState; }
    bool ledRState() const { return m_ledRState; }
    bool ledYState() const { return m_ledYState; }
    bool ledGState() const { return m_ledGState; }
    
    QString oledText() const { return m_oledText; }
    int oledTextSize() const { return m_oledTextSize; }

    int ldrValue() const { return m_ldrValue; }

    float tempValue() const { return m_tempValue; }
    int rhValue() const { return m_rhValue; }
    int pressureValue() const { return m_pressureValue; }
    
    // Mètodes invocables des de QML
    Q_INVOKABLE void connectToBroker();
    Q_INVOKABLE void disconnectFromBroker();
    Q_INVOKABLE void publishButtonPressed(const QString &buttonName);
    Q_INVOKABLE void publishButtonReleased(const QString &buttonName);
    Q_INVOKABLE void setLdrValue(int value);
    Q_INVOKABLE void setTempValue(float value);
    Q_INVOKABLE void setRhValue(int value);
    Q_INVOKABLE void setPressureValue(int value);
    Q_INVOKABLE void publishLdrValue(int value);
    Q_INVOKABLE void publishTempValue(float value);
    Q_INVOKABLE void publishRhValue(int value);
    Q_INVOKABLE void publishPressureValue(int value);
    Q_INVOKABLE void setButtonState(const QString &buttonName, bool pressed);

signals:
    void connectionStateChanged();
    void connectionTextChanged();
    void connectionColorChanged();
    
    void ledWStateChanged();
    void ledRStateChanged();
    void ledYStateChanged();
    void ledGStateChanged();
    
    // Signals OLED
    void oledTextChanged();
    void oledTextSizeChanged();
    void oled10pxReceived(QString text);
    void oled16pxReceived(QString text);
    void oled24pxReceived(QString text);

    // Signals LDR
    void ldrValueChanged();
    void ldrRequested();

    // Signals BME280
    void tempValueChanged();
    void rhValueChanged();
    void pressureValueChanged();
    void tempRequested();
    void rhRequested();
    void pressureRequested();

private slots:
    void onMqttConnected();
    void onMqttConnecting();
    void onMqttDisconnected();
    void onMqttMessageReceived(QString topic, QString message);
    void subscribeToAllTopics();

private:
    MqttBasic *m_mqtt;
    SettingsManager *m_settings;
    
    ConnectionState m_connectionState;
    
    bool m_ledWState;
    bool m_ledRState;
    bool m_ledYState;
    bool m_ledGState;
    
    QString m_oledText;
    int m_oledTextSize;

    int m_ldrValue;

    float m_tempValue;
    int m_rhValue;
    int m_pressureValue;
    
    void setConnectionState(ConnectionState state);
    void updateLedState(const QString &ledName, bool state);

    bool m_btIO0State;
    bool m_btI34State;
    bool m_btI35State;

    void parseJsonLeds(const QString &jsonString);
    QString buildJsonBme() const;
    QString buildJsonIn() const;
};

#endif // MQTTCONTROLLER_H
