#include "mqttcontroller.h"
#include <QDebug>
#include <QTimer>
#include <QJsonDocument>
#include <QJsonObject>
#include <QJsonValue>

MqttController::MqttController(SettingsManager *settings, QObject *parent)
    : QObject(parent)
    , m_settings(settings)
    , m_connectionState(Disconnected)
    , m_ledWState(false)
    , m_ledRState(false)
    , m_ledYState(false)
    , m_ledGState(false)
    , m_oledText("")
    , m_oledTextSize(16)
    , m_ldrValue(0)
    , m_tempValue(20.0)
    , m_rhValue(50)
    , m_pressureValue(1010)
    , m_btIO0State(false)
    , m_btI34State(false)
    , m_btI35State(false)
{
    m_mqtt = new MqttBasic(this);
    
    // Connectar signals de MqttBasic
    connect(m_mqtt, &MqttBasic::vSignalMqttConnected, this, &MqttController::onMqttConnected);
    connect(m_mqtt, &MqttBasic::vSignalMqttConnecting, this, &MqttController::onMqttConnecting);
    connect(m_mqtt, &MqttBasic::vSignalMqttDisconnected, this, &MqttController::onMqttDisconnected);
    connect(m_mqtt, &MqttBasic::vSignalMqttMsgRcv, this, &MqttController::onMqttMessageReceived);
    
    // Auto-connexió si hi ha configuració
    if (!m_settings->isFirstRun() && !m_settings->mqttBroker().isEmpty()) {
        qDebug() << "Auto-connectant al broker MQTT...";
        QTimer::singleShot(500, this, &MqttController::connectToBroker);
    }
}

MqttController::~MqttController()
{
    if (m_mqtt) {
        m_mqtt->vDisconnectBroker();
        delete m_mqtt;
    }
}

QString MqttController::connectionText() const
{
    switch (m_connectionState) {
        case Disconnected: return "Desconnectat";
        case Connecting: return "Connectant...";
        case Connected: return "Connectat";
        default: return "Desconegut";
    }
}

QString MqttController::connectionColor() const
{
    switch (m_connectionState) {
        case Disconnected: return "#E53935";  // Vermell
        case Connecting: return "#1E88E5";    // Blau
        case Connected: return "#43A047";     // Verd
        default: return "#757575";            // Gris
    }
}

void MqttController::connectToBroker()
{
    QString broker = m_settings->mqttBroker();
    QString user = m_settings->mqttUser();
    QString password = m_settings->mqttPassword();
    QString mac = m_settings->mqttMac();
    
    qDebug() << "Connectant a broker:" << broker << "amb MAC:" << mac;
    
    if (broker.isEmpty() || mac.isEmpty()) {
        qDebug() << "Error: Broker o MAC buits. Configura primer.";
        return;
    }
    
    m_mqtt->vSetWsBroker(broker, user, password, mac);
}

void MqttController::disconnectFromBroker()
{
    qDebug() << "Desconnectant del broker MQTT...";
    m_mqtt->vDisconnectBroker();
}

void MqttController::publishButtonPressed(const QString &buttonName)
{
    QString mac = m_settings->mqttMac();
    QString topic = "/" + mac + "/" + buttonName;
    QString message = buttonName + " pressed";

    if (m_connectionState != Connected) {
        qDebug() << "ERROR: Intent de publicar però no connectat!";
        return;
    }
    
    qDebug() << "Publicant:" << topic << "->" << message;
    m_mqtt->vPublish(topic, message);
}

void MqttController::publishButtonReleased(const QString &buttonName)
{
    QString mac = m_settings->mqttMac();
    QString topic = "/" + mac + "/" + buttonName;
    QString message = buttonName + " released";
    
    qDebug() << "Publicant:" << topic << "->" << message;
    m_mqtt->vPublish(topic, message);
}

void MqttController::setConnectionState(ConnectionState state)
{
    if (m_connectionState != state) {
        m_connectionState = state;
        emit connectionStateChanged();
        emit connectionTextChanged();
        emit connectionColorChanged();
    }
}

void MqttController::updateLedState(const QString &ledName, bool state)
{
    if (ledName == "ledW" && m_ledWState != state) {
        m_ledWState = state;
        emit ledWStateChanged();
    } else if (ledName == "ledR" && m_ledRState != state) {
        m_ledRState = state;
        emit ledRStateChanged();
    } else if (ledName == "ledY" && m_ledYState != state) {
        m_ledYState = state;
        emit ledYStateChanged();
    } else if (ledName == "ledG" && m_ledGState != state) {
        m_ledGState = state;
        emit ledGStateChanged();
    }
}

void MqttController::onMqttConnected()
{
    qDebug() << "MQTT Connectat!";
    setConnectionState(Connected);
    subscribeToAllTopics();
}

void MqttController::onMqttConnecting()
{
    qDebug() << "MQTT Connectant...";
    setConnectionState(Connecting);
}

void MqttController::onMqttDisconnected()
{
    qDebug() << "MQTT Desconnectat";
    setConnectionState(Disconnected);
}

void MqttController::subscribeToAllTopics()
{
    QString mac = m_settings->mqttMac();
    
    QString topicLedW = "/" + mac + "/ledW";
    QString topicLedR = "/" + mac + "/ledR";
    QString topicLedY = "/" + mac + "/ledY";
    QString topicLedG = "/" + mac + "/ledG";
    
    m_mqtt->vTopicSubscription(topicLedW);
    m_mqtt->vTopicSubscription(topicLedR);
    m_mqtt->vTopicSubscription(topicLedY);
    m_mqtt->vTopicSubscription(topicLedG);
    
    qDebug() << "Subscrit als temes:" << topicLedW << topicLedR << topicLedY << topicLedG;
    
    // Subscripcions OLED
    QString topic10px = "/" + mac + "/10pxTxt";
    QString topic16px = "/" + mac + "/16pxTxt";
    QString topic24px = "/" + mac + "/24pxTxt";
    
    m_mqtt->vTopicSubscription(topic10px);
    m_mqtt->vTopicSubscription(topic16px);
    m_mqtt->vTopicSubscription(topic24px);
    
    qDebug() << "Subscrit als temes OLED:" << topic10px << topic16px << topic24px;

    // Subscripció LDR request
    QString topicLdrReq = "/" + mac + "/ldrReq";
    m_mqtt->vTopicSubscription(topicLdrReq);
    qDebug() << "Subscrit al tema LDR:" << topicLdrReq;

    // Subscripció a les peticions de dades del BME280
    QString topicTempReq = "/" + mac + "/tempReq";
    QString topicRhReq = "/" + mac + "/rhReq";
    QString topicPressureReq = "/" + mac + "/pressureReq";

    m_mqtt->vTopicSubscription(topicTempReq);
    m_mqtt->vTopicSubscription(topicRhReq);
    m_mqtt->vTopicSubscription(topicPressureReq);

    qDebug() << "Subscrit als temes BME280:" << topicTempReq << topicRhReq << topicPressureReq;

    QString topicJsonBmeReq = "/" + mac + "/jsonBmeReq";
    QString topicJsonInReq = "/" + mac + "/jsonInReq";
    QString topicJsonLeds = "/" + mac + "/jsonLeds";

    m_mqtt->vTopicSubscription(topicJsonBmeReq);
    m_mqtt->vTopicSubscription(topicJsonInReq);
    m_mqtt->vTopicSubscription(topicJsonLeds);

    qDebug() << "Subscrit als temes JSON:" << topicJsonBmeReq << topicJsonInReq << topicJsonLeds;
}

void MqttController::onMqttMessageReceived(QString topic, QString message)
{
    qDebug() << "Missatge rebut - Topic:" << topic << "Message:" << message;
    
    // Extreure el nom del LED del topic (últim segment després de /)
    QStringList parts = topic.split('/');
    if (parts.size() < 2) return;
    
    QString lastPart = parts.last();
    
    // Gestió de LEDs
    if (lastPart.startsWith("led")) {
        bool state = (message == "1");
        updateLedState(lastPart, state);
    }
    // Gestió de missatges OLED
    else if (lastPart == "10pxTxt") {
        m_oledText = message;
        m_oledTextSize = 10;
        emit oledTextChanged();
        emit oledTextSizeChanged();
        emit oled10pxReceived(message);
        qDebug() << "OLED 10px:" << message;
    }
    else if (lastPart == "16pxTxt") {
        m_oledText = message;
        m_oledTextSize = 16;
        emit oledTextChanged();
        emit oledTextSizeChanged();
        emit oled16pxReceived(message);
        qDebug() << "OLED 16px:" << message;
    }
    else if (lastPart == "24pxTxt") {
        m_oledText = message;
        m_oledTextSize = 24;
        emit oledTextChanged();
        emit oledTextSizeChanged();
        emit oled24pxReceived(message);
        qDebug() << "OLED 24px:" << message;
    }
    // Gestió de peticions LDR
    else if (lastPart == "ldrReq") {
        emit ldrRequested();
        qDebug() << "Petició LDR rebuda";
    }else if (lastPart == "tempReq") {
        emit tempRequested();
    }
    else if (lastPart == "rhReq") {
        emit rhRequested();
    }
    else if (lastPart == "pressureReq") {
        emit pressureRequested();
    }
    else if (lastPart == "jsonBmeReq") {
        QString json = buildJsonBme();
        QString topic = "/" + m_settings->mqttMac() + "/jsonBme";
        m_mqtt->vPublish(topic, json);
        qDebug() << "Publicant jsonBme:" << json;
    }
    else if (lastPart == "jsonInReq") {
        QString json = buildJsonIn();
        QString topic = "/" + m_settings->mqttMac() + "/jsonIn";
        m_mqtt->vPublish(topic, json);
        qDebug() << "Publicant jsonIn:" << json;
    }
    else if (lastPart == "jsonLeds") {
        parseJsonLeds(message);
    }
}

void MqttController::setLdrValue(int value)
{
    m_ldrValue = value;
    emit ldrValueChanged();
}

void MqttController::publishLdrValue(int value)
{
    QString mac = m_settings->mqttMac();
    QString topic = "/" + mac + "/ldr";
    QString message = QString::number(value);

    qDebug() << "Publicant LDR:" << topic << "->" << message;
    m_mqtt->vPublish(topic, message);

    setLdrValue(value);
}

void MqttController::setTempValue(float value)
{
    m_tempValue = value;
    emit tempValueChanged();
}

void MqttController::publishTempValue(float value)
{
    QString mac = m_settings->mqttMac();
    QString topic = "/" + mac + "/temp";
    QString message = QString::number(value, 'f', 1);  // 1 decimal

    m_mqtt->vPublish(topic, message);
    setTempValue(value);
}

void MqttController::setRhValue(int value)
{
    m_rhValue = value;
    emit rhValueChanged();
}

void MqttController::publishRhValue(int value)
{
    QString mac = m_settings->mqttMac();
    QString topic = "/" + mac + "/rh";
    QString message = QString::number(value);

    m_mqtt->vPublish(topic, message);
    setRhValue(value);
}

void MqttController::setPressureValue(int value)
{
    m_pressureValue = value;
    emit pressureValueChanged();
}

void MqttController::publishPressureValue(int value)
{
    QString mac = m_settings->mqttMac();
    QString topic = "/" + mac + "/pressure";
    QString message = QString::number(value);

    m_mqtt->vPublish(topic, message);
    setPressureValue(value);
}

void MqttController::setButtonState(const QString &buttonName, bool pressed)
{
    if (buttonName == "btIO0") m_btIO0State = pressed;
    else if (buttonName == "btI34") m_btI34State = pressed;
    else if (buttonName == "btI35") m_btI35State = pressed;
}

void MqttController::parseJsonLeds(const QString &jsonString)
{
    QJsonDocument doc = QJsonDocument::fromJson(jsonString.toUtf8());
    if (doc.isNull() || !doc.isObject()) {
        qDebug() << "Error: JSON invàlid";
        return;
    }

    QJsonObject obj = doc.object();

    auto processLed = [&](const QString &key) {
        if (obj.contains(key)) {
            QJsonValue val = obj[key];
            bool state = false;
            if (val.isBool()) state = val.toBool();
            else if (val.isDouble()) state = (val.toInt() == 1);
            else if (val.isString()) state = (val.toString() == "1");
            updateLedState(key, state);
        }
    };

    processLed("ledW");
    processLed("ledR");
    processLed("ledY");
    processLed("ledG");
}

QString MqttController::buildJsonBme() const
{
    QJsonObject obj;
    obj["Tx100"] = qRound(m_tempValue * 100);
    obj["RHx100"] = m_rhValue * 100;
    obj["Px100"] = m_pressureValue * 100;

    QJsonDocument doc(obj);
    return QString::fromUtf8(doc.toJson(QJsonDocument::Compact));
}

QString MqttController::buildJsonIn() const
{
    QJsonObject obj;
    obj["Tx100"] = qRound(m_tempValue * 100);
    obj["RHx100"] = m_rhValue * 100;
    obj["Px100"] = m_pressureValue * 100;
    obj["bIO0"] = m_btIO0State ? 1 : 0;
    obj["bI34"] = m_btI34State ? 1 : 0;
    obj["bI35"] = m_btI35State ? 1 : 0;
    obj["LDR"] = m_ldrValue;

    QJsonDocument doc(obj);
    return QString::fromUtf8(doc.toJson(QJsonDocument::Compact));
}
