import QtQuick
import Qt5Compat.GraphicalEffects

Item {
    id: root
    
    // Propietats per posicionament i mida en píxels de la imatge original
    property real boardX: 0
    property real boardY: 0
    property real boardWidth: 128  // Amplada OLED real en píxels
    property real boardHeight: 64   // Alçada OLED real en píxels
    
    // Propietats d'escala i offset
    property real imageScale: 1.0
    property real xOffset: 0
    property real yOffset: 0
    
    // Propietats de visualització
    property string displayText: settingsManager ? settingsManager.mqttMac : ""
    property int textSize: 16       // 10, 16, o 24
    property int textX: 0           // Coordenada X en píxels OLED
    property int textY: 24          // Coordenada Y en píxels OLED (centrat per 16px)
    
    // Constants OLED real
    readonly property int oledWidthPx: 128
    readonly property int oledHeightPx: 64
    
    // Mides escalades dinàmicament
    width: boardWidth * imageScale
    height: boardHeight * imageScale
    
    // Posicions escalades dinàmicament
    x: boardX * imageScale + xOffset
    y: boardY * imageScale + yOffset
    
    // Funció per calcular la mida de font proporcionalment
    function calculateFontSize(oledPixels) {
        const scaleFactor = boardWidth / oledWidthPx
        return oledPixels * scaleFactor * imageScale
    }
    
    // Funció per actualitzar coordenada Y segons mida de text (centrat vertical)
    function updateTextY(size) {
        // Centrar verticalment segons la mida del text
        return Math.round((oledHeightPx - size) / 2)
    }
    
    // Rectangle que simula la pantalla OLED
    Rectangle {
        id: oledScreen
        anchors.fill: parent
        color: "#000000"  // Negre
        border.color: "#4000FFFF"  // Cyan
        border.width: 1 * imageScale
        
        // Àrea de text
        Item {
            id: textArea
            anchors.fill: parent
            anchors.margins: 2 * imageScale  // Petit marge
            clip: true  // No dibuixa fora del rectangle
            
            Text {
                id: displayedText
                x: 5+root.textX * (root.boardWidth / root.oledWidthPx) * root.imageScale
                y: root.textY * (root.boardHeight / root.oledHeightPx) * root.imageScale
                
                text: root.displayText
                color: "#00FFFF"  // Cyan (color típic OLED)
                font.family: "Arial"
                font.pixelSize: root.calculateFontSize(root.textSize)
                font.weight: Font.Normal
                
                // Retalla el text si no cap
                elide: Text.ElideRight
                width: textArea.width - x

                layer.enabled: true
                // layer.effect: DropShadow {
                //     horizontalOffset: 0
                //     verticalOffset: 0
                //     radius: 8
                //     samples: 17
                //     color: "#80FFFF"  // Cyan semitransparent
                // }
                layer.effect: Glow {
                    samples: 9
                    color: "#00FFFF"
                    spread: 0.2
                    radius: 4
                }
            }
        }
    }
    
    // Connections amb MqttController per rebre missatges OLED
    Connections {
        target: mqttController
        
        function onOled10pxReceived(text) {
            root.displayText = text
            root.textSize = 10
            root.textY = root.updateTextY(10)
        }
        
        function onOled16pxReceived(text) {
            root.displayText = text
            root.textSize = 16
            root.textY = root.updateTextY(16)
        }
        
        function onOled24pxReceived(text) {
            root.displayText = text
            root.textSize = 24
            root.textY = root.updateTextY(24)
        }
    }
}
