import QtQuick
import QtQuick.Controls

Item {
    id: root
    
    // Propietats per posicionament i mida en píxels de la imatge original
    property real boardX: 0
    property real boardY: 0
    property real boardWidth: 30
    property real boardHeight: 30
    
    // Propietats d'escala i offset
    property real imageScale: 1.0
    property real xOffset: 0
    property real yOffset: 0
    
    // Valor del sensor LDR (0-4095)
    property int currentValue: 500
    
    // Mides escalades dinàmicament
    width: boardWidth * imageScale
    height: boardHeight * imageScale
    
    // Posicions escalades dinàmicament
    x: boardX * imageScale + xOffset
    y: boardY * imageScale + yOffset
    
    // Timer per generar valors aleatoris (si està activat)
    Timer {
        id: randomTimer
        interval: 2000  // Cada 2 segons
        running: settingsManager ? settingsManager.ldrRandom : false
        repeat: true
        
        onTriggered: {
            if (!settingsManager) return

            // Generar valor aleatori entre ldrMin i ldrMax
            var min = settingsManager.ldrMin
            var max = settingsManager.ldrMax
            root.currentValue = Math.floor(Math.random() * (max - min + 1)) + min
        }
    }
    
    // Connections amb MqttController per respondre a peticions
    Connections {
        target: mqttController
        
        function onLdrRequested() {
            // Quan arriba una petició, publicar el valor actual
            mqttController.publishLdrValue(root.currentValue)
            console.log("LDR request rebut, publicant valor:", root.currentValue)
        }
    }
    
    // Indicador visual del sensor LDR
    Rectangle {
        id: ldrIndicator
        anchors.fill: parent
        color: "#40808080"
        radius: Math.min(width, height) / 2
        border.color: "gray"
        border.width: 2 * imageScale
        
        Text {
            anchors.centerIn: parent
            text: "LDR"
            color: "white"
            // font.pixelSize: 8 * imageScale
            font.pixelSize: 30 * imageScale
            font.bold: false
        }
        
        MouseArea {
            anchors.fill: parent
            onClicked: {
                if (settingsManager && !settingsManager.ldrRandom) {
                    sliderPopup.open()
                }
            }
        }
    }
    
    // Popup amb slider (només visible si NO és aleatori)
    Popup {
        id: sliderPopup
        // x: parent.width + 10
        x: parent.width - 80
        y: -150 // -60
        width: 250 // 280
        height: 140
        modal: false
        focus: true
        closePolicy: Popup.CloseOnEscape | Popup.CloseOnPressOutside
        visible: false
        
        Column {
            anchors.fill: parent
            anchors.margins: 10
            spacing: 15
            
            Text {
                text: "Valor LDR: " + root.currentValue
                font.pixelSize: 16
                font.bold: true
            }
            
            Slider {
                id: valueSlider
                width: parent.width
                from: 0
                to: 4095
                stepSize: 1
                value: root.currentValue
                
                onValueChanged: {
                    root.currentValue = Math.round(value)
                }
            }
            
            // Row {
            //     spacing: 10
            //     width: parent.width
                
            //     Button {
            //         text: "Tanca"
            //         onClicked: sliderPopup.close()
            //     }
                
            //     Button {
            //         text: "Publica valor"
            //         highlighted: true
            //         onClicked: {
            //             mqttController.publishLdrValue(root.currentValue)
            //             sliderPopup.close()
            //         }
            //     }
            // }
        }
    }
    
    // Visualització del valor (sempre visible, a la dreta del sensor)
    Rectangle {
        id: valueDisplay
        property alias imagescale: root.imageScale
        // x: parent.width + (5 * imageScale)
        // y: (parent.height - height) / 2
        x: (parent.width - width) / 2
        y: parent.height + (10 * imagescale)
        width: valueText.width + (10 * imageScale)
        height: valueText.height + (6 * imageScale)
        color: "#E0000000"
        radius: 3 * imageScale
        border.color: "#808080"
        border.width: 1 * imageScale
        visible: settingsManager && (settingsManager.ldrRandom || root.currentValue > 0)

        Text {
            id: valueText
            anchors.centerIn: parent
            text: root.currentValue.toString()
            color: "white"
            // font.pixelSize: 12 * imageScale
            font.pixelSize: 30 * imageScale
            font.family: "Monospace"
        }
    }
}
