import QtQuick
import QtQuick.Controls

Item {
    id: root
    
    // Propietats per posicionament i mida en píxels de la imatge original
    property real boardX: 0
    property real boardY: 0
    property real boardWidth: 40
    property real boardHeight: 40
    
    // Propietats d'escala i offset
    property real imageScale: 1.0
    property real xOffset: 0
    property real yOffset: 0
    
    // Valors dels sensors BME280
    property real tempValue: 20.0      // °C (amb 1 decimal)
    property int rhValue: 50           // % (sense decimals)
    property int pressureValue: 1010   // hPa (sense decimals)
    
    // Mides escalades dinàmicament
    width: boardWidth * imageScale
    height: boardHeight * imageScale
    
    // Posicions escalades dinàmicament
    x: boardX * imageScale + xOffset
    y: boardY * imageScale + yOffset
    
    // Timer per generar valors aleatoris (si està activat)
    Timer {
        id: randomTimer
        interval: 2000  // Cada 2 segons
        running: settingsManager ? settingsManager.bmeRandom : false
        repeat: true
        
        onTriggered: {
            if (!settingsManager) return

            // Generar valors aleatoris entre els rangs configurats
            var tempMin = settingsManager.tempMin
            var tempMax = settingsManager.tempMax
            root.tempValue = Math.random() * (tempMax - tempMin) + tempMin
            
            var rhMin = settingsManager.rhMin
            var rhMax = settingsManager.rhMax
            root.rhValue = Math.floor(Math.random() * (rhMax - rhMin + 1)) + rhMin
            
            var pressureMin = settingsManager.pressureMin
            var pressureMax = settingsManager.pressureMax
            root.pressureValue = Math.floor(Math.random() * (pressureMax - pressureMin + 1)) + pressureMin

            if (mqttController) {
                // mqttController.publishTempValue(root.tempValue)
                // mqttController.publishRhValue(root.rhValue)
                // mqttController.publishPressureValue(root.pressureValue)
                mqttController.setTempValue(root.tempValue)
                mqttController.setRhValue(root.rhValue)
                mqttController.setPressureValue(root.pressureValue)
            }
        }
    }
    
    // Connections amb MqttController per respondre a peticions
    Connections {
        target: mqttController
        
        function onTempRequested() {
            mqttController.publishTempValue(root.tempValue)
            console.log("Temperatura request rebut, publicant valor:", root.tempValue)
        }
        
        function onRhRequested() {
            mqttController.publishRhValue(root.rhValue)
            console.log("Humitat request rebut, publicant valor:", root.rhValue)
        }
        
        function onPressureRequested() {
            mqttController.publishPressureValue(root.pressureValue)
            console.log("Pressió request rebut, publicant valor:", root.pressureValue)
        }
    }
    
    // Indicador visual del sensor BME280
    Rectangle {
        id: bme280Indicator
        anchors.fill: parent
        color: "#4080A0A0"  // Blau clar
        radius: Math.min(width, height) / 2
        border.color: "#6090B0B0"
        border.width: 2 * imageScale
        
        Text {
            anchors.centerIn: parent
            text: "BME\n280"
            color: "white"
            // font.pixelSize: 10 * imageScale
            font.pixelSize: 30 * imageScale
            font.bold: true
            horizontalAlignment: Text.AlignHCenter
        }
        
        MouseArea {
            anchors.fill: parent
            onClicked: {
                if (!settingsManager.bmeRandom) {
                    sliderPopup.open()
                }
            }
        }
    }
    
    // Popup amb 3 sliders (només visible si NO és aleatori)
    Popup {
        id: sliderPopup
        // x: parent.width + 10
        x: parent.width - 150
        y: -80
        // width: 340
        width: 250
        height: 250 // 230
        modal: false
        focus: true
        closePolicy: Popup.CloseOnEscape | Popup.CloseOnPressOutside
        visible: false
        
        Column {
            anchors.fill: parent
            anchors.margins: 10
            spacing: 10
            
            // Temperatura
            Column {
                width: parent.width
                spacing: 5
                
                Text {
                    text: "Temperatura: " + root.tempValue.toFixed(1) + "°C"
                    font.pixelSize: 14
                    font.bold: true
                }
                
                Slider {
                    id: tempSlider
                    width: parent.width
                    from: -40
                    to: 85
                    stepSize: 0.1
                    value: root.tempValue
                    
                    onMoved: {
                        root.tempValue = Math.round(value * 10) / 10  // 1 decimal
                        if (mqttController) {
                            mqttController.setTempValue(root.tempValue)
                        }
                    }
                }
            }
            
            // Humitat relativa
            Column {
                width: parent.width
                spacing: 5
                
                Text {
                    text: "Humitat relativa: " + root.rhValue + "%"
                    font.pixelSize: 14
                    font.bold: true
                }
                
                Slider {
                    id: rhSlider
                    width: parent.width
                    from: 0
                    to: 100
                    stepSize: 1
                    value: root.rhValue
                    
                    onMoved: {
                        root.rhValue = Math.round(value)
                        if (mqttController) {
                            mqttController.setRhValue(root.rhValue)
                        }
                    }
                }
            }
            
            // Pressió atmosfèrica
            Column {
                width: parent.width
                spacing: 5
                
                Text {
                    text: "Pressió atmosfèrica: " + root.pressureValue + " hPa"
                    font.pixelSize: 14
                    font.bold: true
                }
                
                Slider {
                    id: pressureSlider
                    width: parent.width
                    from: 300
                    to: 1100
                    stepSize: 1
                    value: root.pressureValue
                    
                    onMoved: {
                        root.pressureValue = Math.round(value)
                        if (mqttController) {
                            mqttController.setPressureValue(root.pressureValue)
                        }
                    }
                }
            }
        }
    }
    
    // Visualització dels valors (sempre visible, a sota del sensor)
    Rectangle {
        id: valuesDisplay
        // x: (parent.width - width) / 2  // Centrat horitzontalment
        // y: parent.height + (5 * imageScale)  // A sota del sensor
        x: -(width + (140 * imageScale))  // A l'esquerra amb marge de 5px
        y: (parent.height - height) / 2  // Centrat verticalment
        // width: valuesColumn.width + (10 * imageScale)
        // height: valuesColumn.height + (6 * imageScale)
        width: valuesColumn.width + (40 * imageScale)
        height: valuesColumn.height + (24 * imageScale)
        color: "#E0000000"
        radius: 3 * imageScale
        border.color: "#808080"
        border.width: 1 * imageScale
        visible: settingsManager && (settingsManager.bmeRandom || root.tempValue > 0)

        Column {
            id: valuesColumn
            anchors.centerIn: parent
            spacing: 2 * imageScale
            
            Text {
                text: root.tempValue.toFixed(1) + "°C"
                color: "white"
                // font.pixelSize: 12 * imageScale
                font.pixelSize: 30 * imageScale
                font.family: "Monospace"
                // horizontalAlignment: Text.AlignHCenter
                horizontalAlignment: Text.AlignRight  // Justificat a la dreta dins el Text
            }
            
            Text {
                text: root.rhValue + "%"
                color: "white"
                // font.pixelSize: 12 * imageScale
                font.pixelSize: 30 * imageScale
                font.family: "Monospace"
                horizontalAlignment: Text.AlignHCenter
            }
            
            Text {
                text: root.pressureValue + " hPa"
                color: "white"
                // font.pixelSize: 12 * imageScale
                font.pixelSize: 30 * imageScale
                font.family: "Monospace"
                horizontalAlignment: Text.AlignHCenter
            }
        }
    }
}
