//The BlueDot BME680 library is being conceived to read the BME680 sensor with an Arduino Uno
//This library is an on going development and this version is UNFINISHED (Status 2017)

//So far I've managed to read temperature, pressure and humidity from the BME680 sensor with similar results as from a BME280 sensor
//The BME680 seems to show a noticeable self-heating effect, basically heating the board where it is mounted
//Temperature measurements are a bit misleading, because the board becomes a bit hotter than the air around it
//Bosch Sensortec acknowledges this fact and recomends that the sensor should not run continuously
//In practice, I suppose measuring once every couple seconds should help to minimize the self-heating effect
//At the BSEC Integration Guide (BST-BME680-AN008-34) Bosch goes further and advises to compensate both temperature and humidity readings caused by the integrated hot plate (page 37)

//Furthermore With this library you can read the resistance (in ohms) of the gas sensitive layer

//Finally, this library only uses the I2C bus
//The implementation of the SPI communication is still unfinished

//Until this library is working properly, I hope this unfinished version can help anyone to understand how this sensor works


#if defined(_AVR_)
#include <util/delay.h>
#endif

#include "BlueDot_BME680.h"
#include "Wire.h"
#include "SPI.h"

//##########################################################################
//Data arrays for conversion of raw gas measurements into resistance
//These values are used for calculating the gas resistance (see Datasheet, page 20, table 12)
//Values are called by function "readGas(void)"

static float const_array1[16] = {1, 1, 1, 1, 1, 0.99, 1, 0.992, 1, 1, 0.998, 0.995, 1, 0.99, 1, 1};
static double const_array2[16] = {8000000.0, 4000000.0, 2000000.0, 1000000.0, 499500.4995, 248262.1648, 125000.0, 
63004.03226, 31281.28128, 15625.0, 7812.5, 3906.25, 1953.125, 976.5625, 488.28125, 244.140625};


//##########################################################################
BlueDot_BME680::BlueDot_BME680()
{
	parameter.communication;
	parameter.I2CAddress;
	parameter.sensorMode;
	parameter.IIRfilter;
	parameter.tempOversampling;
	parameter.pressOversampling;
	parameter.humidOversampling;
	parameter.pressureSeaLevel = 1013.25;

}
//##########################################################################
//INITIALIZATION
//##########################################################################
//So far the initialization of the BME680 consists on initializing the I2C bus, setting the IIR filter, reading the calibration coefficients and checking the Chip ID
//This is enough to read temperature, humidity and pressure values
//In order to read the gas sensor, we also need to set the hot plate time (e.g. 100 ms), set the hot plate target temperature (e.g. 320 °C) and select the hot plate profile


uint8_t BlueDot_BME680::init(void)
{
	
	Wire.begin();
	writeByte(BME680_CTRL_GAS_1, 0);
	writeIIRFilter();							//set IIR Filter coefficient
	readCoefficients();							//read coefficients from calibration registers
	
	//The following three functions enable gas sensor
	//If you wish to turn the gas sensor off (and reduce the sensor's self-heating), just comment out the following functions
	
	calculateHotPlateTime();					//set heat up duration to 100 ms			
	calculateHotPlateRes();						//convert target_temp to heater resistance (set on Arduino Sketch)
	setHotPlateProfile();						//select heater profile to be used and set run_gas_I to 1 to enable gas measurement
	
	//Now finish the initialization by reading the chip ID
	
	return checkID();							//read chip ID

}
//##########################################################################
//SET UP FUNCTIONS
//##########################################################################
//This function reads the BME680_CHIP_ID register and returns the value of the Chip ID.
//All BME680 chips have the same ID: 0x61
//Please note that the Chip ID from the BME280 sensor is 0x60

uint8_t BlueDot_BME680::checkID(void)
{
	uint8_t chipID;
	chipID = readByte(BME680_CHIP_ID);
	return chipID;	
}
//##########################################################################
//Here we read the coefficients from the calibration registers
//These functions were taken directly from the file "bme680.c" from Bosch (Revision 2.2.0 $ from 5 May 2017)

void BlueDot_BME680::readCoefficients(void)
{
	bme680_coefficients.dig_T1 = ((uint16_t)(readByte(BME680_DIG_T1_MSB) << 8) + readByte(BME680_DIG_T1_LSB));
	bme680_coefficients.dig_T2 = ((int16_t)(readByte(BME680_DIG_T2_MSB) << 8) + readByte(BME680_DIG_T2_LSB));
	bme680_coefficients.dig_T3 = ((int8_t)(readByte(BME680_DIG_T3)));
	
	bme680_coefficients.dig_P1 = ((uint16_t)(readByte(BME680_DIG_P1_MSB) << 8) + readByte(BME680_DIG_P1_LSB));
	bme680_coefficients.dig_P2 = ((int16_t)(readByte(BME680_DIG_P2_MSB) << 8) + readByte(BME680_DIG_P2_LSB));
	bme680_coefficients.dig_P3 = ((int8_t)(readByte(BME680_DIG_P3)));
	bme680_coefficients.dig_P4 = ((int16_t)(readByte(BME680_DIG_P4_MSB) << 8) + readByte(BME680_DIG_P4_LSB));
	bme680_coefficients.dig_P5 = ((int16_t)(readByte(BME680_DIG_P5_MSB) << 8) + readByte(BME680_DIG_P5_LSB));
	bme680_coefficients.dig_P6 = ((int8_t)(readByte(BME680_DIG_P6)));
	bme680_coefficients.dig_P7 = ((int8_t)(readByte(BME680_DIG_P7)));
	bme680_coefficients.dig_P8 = ((int16_t)(readByte(BME680_DIG_P8_MSB) << 8) + readByte(BME680_DIG_P8_LSB));
	bme680_coefficients.dig_P9 = ((int16_t)(readByte(BME680_DIG_P9_MSB) << 8) + readByte(BME680_DIG_P9_LSB));
	bme680_coefficients.dig_P10 = ((uint8_t)(readByte(BME680_DIG_P10)));
	
	bme680_coefficients.dig_H1 = (uint16_t)(((uint16_t)(readByte(BME680_DIG_H1_MSB)) << 4) + ((uint8_t)(readByte(BME680_DIG_H1_LSB)) & BME680_BIT_MASK_H1_DATA));
	bme680_coefficients.dig_H2 = (uint16_t)(((uint16_t)(readByte(BME680_DIG_H2_MSB)) << 4) + (((uint8_t)(readByte(BME680_DIG_H2_LSB))) >> 4));
	bme680_coefficients.dig_H3 = (int8_t)(readByte(BME680_DIG_H3));
	bme680_coefficients.dig_H4 = (int8_t)(readByte(BME680_DIG_H4));
	bme680_coefficients.dig_H5 = (int8_t)(readByte(BME680_DIG_H5));
	bme680_coefficients.dig_H6 = (uint8_t)(readByte(BME680_DIG_H6));
	bme680_coefficients.dig_H7 = (int8_t)(readByte(BME680_DIG_H7));

//The following calibration parameters are used to convert the target temperature of the hot plate to a register value
//This register value is written to the registers res_heat_x<7:0> (see Datasheet, "Gas sensor heating and measurement", pages 18 and 19) 
	
	bme680_coefficients.par_g1 = (int8_t)(readByte(BME680_DIG_G1));	
	bme680_coefficients.par_g2 = (int16_t)(readByte(BME680_DIG_G2_MSB) << 8) + readByte(BME680_DIG_G2_LSB);	
	bme680_coefficients.par_g3 = (int8_t)(readByte(BME680_DIG_G3));
	
	bme680_coefficients.res_heat_range = (int8_t)(readByte(BME680_RES_HEAT_RG));
	bme680_coefficients.res_heat_val   = (int8_t)(readByte(BME680_RES_HEAT_VL));	
}

//##########################################################################
//We set up the IIR Filter through bits 4, 3 and 2 from Config Register (0x75)]
//The other bits from this register won't be used in this program and remain 0
//Please refer to the BME680 Datasheet for more information (page 28)


void BlueDot_BME680::writeIIRFilter(void)
{	
	byte value;
	value = (parameter.IIRfilter << 2) & 0b00011100;
	writeByte(BME680_CONFIG, value);
}
//##########################################################################
//Here we set the oversampling for humidity measurements by writting to the register BME680_CTRL_HUM 
//We also set the oversampling factors for measuring temperature and pressure by writting to the register BME680_CTRL_MEAS
//For that, refer to the Datasheet, "Temperature, pressure and relative humidity control registers", pages 27 and 28)
//Finally, here we define the sensor mode by writting to the register BME680_CTRL_MEAS (bits 1 and 0)
//Please note that setting the sensor mode to "01" (Forced Mode) starts a measurement
//After a single measurement the sensor will return to sensor mode "00" (Sleep Mode)
//Therefore we must set the sensor mode to Forced Mode on the Loop function 


void BlueDot_BME680::writeCTRLMeas(void)
{
	byte value;
	value = parameter.humidOversampling & 0b00000111;
	writeByte(BME680_CTRL_HUM, value);
	
	value = (parameter.tempOversampling << 5) & 0b11100000;
	value |= (parameter.pressOversampling << 2) & 0b00011100;
	value |= parameter.sensorMode & 0b00000011;
	writeByte(BME680_CTRL_MEAS, value);		
}
//##########################################################################
//DATA READOUT FUNCTIONS
//##########################################################################
//Reading the temperature value from the BME680 is identical to reading the temperature from the BME280
//If the oversampling factor is set to 0, the temperature measurement is disabled
//This function returns the temperature in Celsius


float BlueDot_BME680::readTempC(void)
{	
	if (parameter.tempOversampling == 0b000)					//disabling the temperature measurement function
	{
		return 0;
	}
	
	else
	{
		int32_t adc_T;
		adc_T = (uint32_t)readByte(BME680_TEMPERATURE_MSB) << 12;
		adc_T |= (uint32_t)readByte(BME680_TEMPERATURE_LSB) << 4;
		adc_T |= (readByte(BME680_TEMPERATURE_XLSB) >> 4) & 0b00001111;
		
		int64_t var1, var2;
		
		var1 = ((((adc_T>>3) - ((int32_t)bme680_coefficients.dig_T1<<1))) * ((int32_t)bme680_coefficients.dig_T2)) >> 11;
		var2 = (((((adc_T>>4) - ((int32_t)bme680_coefficients.dig_T1)) * ((adc_T>>4) - ((int32_t)bme680_coefficients.dig_T1))) >> 12) *
		((int32_t)bme680_coefficients.dig_T3)) >> 14;
		t_fine = var1 + var2;
		float T = (t_fine * 5 + 128) >> 8;
		T = T / 100;
		return T;
	}
}

//##########################################################################
//This function also reads the temperature from BME680 and converts the resulting value from Celsius to Fahrenheit


float BlueDot_BME680::readTempF(void)
{	
	if (parameter.tempOversampling == 0b000)					//disabling the temperature measurement function
	{
		return 0;
	}
	
	else
	{
		int32_t adc_T;
		adc_T = (uint32_t)readByte(BME680_TEMPERATURE_MSB) << 12;
		adc_T |= (uint32_t)readByte(BME680_TEMPERATURE_LSB) << 4;
		adc_T |= (readByte(BME680_TEMPERATURE_XLSB) >> 4) & 0b00001111;
		
		int64_t var1, var2;
		
		var1 = ((((adc_T>>3) - ((int32_t)bme680_coefficients.dig_T1<<1))) * ((int32_t)bme680_coefficients.dig_T2)) >> 11;
		var2 = (((((adc_T>>4) - ((int32_t)bme680_coefficients.dig_T1)) * ((adc_T>>4) - ((int32_t)bme680_coefficients.dig_T1))) >> 12) *
		((int32_t)bme680_coefficients.dig_T3)) >> 14;
		t_fine = var1 + var2;
		float T = (t_fine * 5 + 128) >> 8;
		T = T / 100;
		T = (T * 1.8) + 32;
		return T;
	}
}
//##########################################################################
//Temperature in Kelvin is needed for the conversion of pressure to altitude	
//Both parameters "tempOutsideCelsius" and "tempOutsideFahrenheit" are set to 999 as default (see .h file)
//If the user chooses to read temperature in Celsius, "tempOutsideFahrenheit" remains 999
//If the user chooses to read temperature in Fahrenheit instead, "tempOutsideCelsius" remains 999
//If both values are used, then the temperature in Celsius will be used for the conversion
//If none of them are used, then the default value of 288.15 will be used (i.e. 273.15 + 15)


float BlueDot_BME680::convertTempKelvin(void)
{
	float tempOutsideKelvin;	
	
	if (parameter.tempOutsideCelsius != 999 & parameter.tempOutsideFahrenheit == 999 )   
	{
		tempOutsideKelvin = parameter.tempOutsideCelsius;
		tempOutsideKelvin = tempOutsideKelvin + 273.15;
		return tempOutsideKelvin;		
	}
	
	if (parameter.tempOutsideCelsius != 999 & parameter.tempOutsideFahrenheit != 999 )   
	{
		tempOutsideKelvin = parameter.tempOutsideCelsius;
		tempOutsideKelvin = tempOutsideKelvin + 273.15;
		return tempOutsideKelvin;		
	}
	
	if (parameter.tempOutsideFahrenheit != 999 & parameter.tempOutsideCelsius == 999)
	{
		
		tempOutsideKelvin = (parameter.tempOutsideFahrenheit - 32);
		tempOutsideKelvin = tempOutsideKelvin * 5;
		tempOutsideKelvin = tempOutsideKelvin / 9;
		tempOutsideKelvin = tempOutsideKelvin + 273.15;
		return tempOutsideKelvin;	
	}
	
	if (parameter.tempOutsideFahrenheit == 999 & parameter.tempOutsideCelsius == 999)
	{
		tempOutsideKelvin = 273.15 + 15;
		return tempOutsideKelvin; 
	}
	
	tempOutsideKelvin = 273.15 + 15;
	return tempOutsideKelvin;

}
//##########################################################################
//Reading the pressure from the BME680 is NOT the same as reading the pressure from the BME280 sensor
//The following functions were taken from the file "bme680_calculations.c" from Bosch Sensortec (Revision 2.2.0 $ from 5 May 2017)
//Note that I chose to let the variable "var4" out of the calculation (see commentary below)


float BlueDot_BME680::readPressure(void)
{
	if (parameter.pressOversampling == 0b000)						//disabling the pressure measurement function
	{
		return 0;
	}
	
	else
	{		
		readTempC();		
		
		uint32_t adc_P;
		adc_P = (uint32_t)readByte(BME680_PRESSURE_MSB) << 12;
		adc_P |= (uint32_t)readByte(BME680_PRESSURE_LSB) << 4;
		adc_P |= (readByte(BME680_PRESSURE_XLSB) >> 4) & 0b00001111;		
		
		int32_t var1 = 0;
		int32_t var2 = 0;
		int32_t var3 = 0;
		//int32_t var4 = 0;
		int32_t P = 0;
		
		var1 = (((int32_t)t_fine) >> 1) - 64000;
		
		var2 = ((((var1 >> 2) * (var1 >> 2)) >> 11) * (int32_t)bme680_coefficients.dig_P6) >> 2;
		var2 = var2 + ((var1 * (int32_t)bme680_coefficients.dig_P5) << 1);
		var2 = (var2 >> 2) + ((int32_t)bme680_coefficients.dig_P4 << 16);
		
		var1 = (((((var1 >> 2) * (var1 >> 2)) >> 13) * ((int32_t)bme680_coefficients.dig_P3 << 5)) >> 3) + (((int32_t)bme680_coefficients.dig_P2 * var1) >> 1);
		var1 = var1 >> 18;
		var1 = ((32768 + var1) * (int32_t)bme680_coefficients.dig_P1) >> 15;
		
		P = 1048576 - adc_P;
		P = (int32_t)((P - (var2 >> 12)) * ((uint32_t)3125));		
	
//This is the original calculation taken from the file "bme680_calculations.c"
//I chose to change part of the calculation, because the values I was getting were way off
	
		/*
		var4 = (1 << 31);
		if (P >= var4)
			P = ((P / (uint32_t)var1) << 1);				
		else
			P = ((P << 1) / (uint32_t)var1);
		*/
		
//See what I did here? This corresponds to the "else" situation

		P = ((P << 1) / (uint32_t)var1);		
		
		var1 = ((int32_t)bme680_coefficients.dig_P9 * (int32_t)(((P >> 3) * (P >> 3)) >> 13)) >> 12;		
		var2 = ((int32_t)(P >> 2) * (int32_t)bme680_coefficients.dig_P8) >> 13;		
		var3 = ((int32_t)(P >> 8) * (int32_t)(P >> 8) * (int32_t)(P >> 8) * (int32_t)bme680_coefficients.dig_P10) >> 17;		
		
		P = (int32_t)(P) + ((var1 + var2 + var3 + ((int32_t)bme680_coefficients.dig_P7 << 7)) >> 4);
		
		return (float)P/100;		
	}
}
//##########################################################################
//Reading the humidity values from the BME680 is DIFFERENT from reading the humidity value from the BME280 sensor
//The following functions were taken from the file "bme680_calculations.c" from Bosch Sensortec (Revision 2.2.0 $ from 5 May 2017)


float BlueDot_BME680::readHumidity(void)
{
	if (parameter.humidOversampling == 0b000)					//disabling the humitidy measurement function
	{
		return 0;
	}
	
	else
	{
		int32_t adc_H;
		adc_H = (uint32_t)readByte(BME680_HUMIDITY_MSB) << 8;
		adc_H |= (uint32_t)readByte(BME680_HUMIDITY_LSB);		
		
		int32_t temp_scaled = 0;
		int32_t var1 = 0;
		int32_t var2 = 0;
		int32_t var3 = 0;
		int32_t var4 = 0;
		int32_t var5 = 0;
		int32_t var6 = 0;
		int32_t H    = 0;
		
		temp_scaled = ((((int32_t)t_fine) * 5) + 128) >> 8;		
	
		var1 = (int32_t)adc_H - ((int32_t)((int32_t)bme680_coefficients.dig_H1 << 4)) - 
			   ((temp_scaled * (int32_t)bme680_coefficients.dig_H3/(int32_t)100) >> 1);
			   
		
		var2 = ((int32_t)bme680_coefficients.dig_H2 * (((temp_scaled * (int32_t)bme680_coefficients.dig_H4)/(int32_t)100) + 
				(((temp_scaled * ((temp_scaled * (int32_t)bme680_coefficients.dig_H5)/((int32_t)100))) >> 6)/((int32_t)100)) + 
				(int32_t)(1 << 14))) >> 10;
				
		var3 = var1 * var2;
		
		var4 = ((((int32_t)bme680_coefficients.dig_H6) << 7) + ((temp_scaled * (int32_t)bme680_coefficients.dig_H7)/((int32_t)100))) >> 4;
		
		var5 = ((var3 >> 14) * (var3 >> 14)) >> 10;
		var6 = (var4 * var5) >> 1;
		
		H = (var3 + var6) >> 12;
		
//This part avoids returning humidity values above 100% and below 0%, which wouldn't make sense		
		if (H > 102400)
			H = 102400;
		else if (H < 0)
			H = 0;
		
		return H/1024.;		
	}
}
//##########################################################################
//To calculate the altitude (in meters) from the pressure (in hPa) I've used the international Barometric Formula
//I've used the formula from the German Wikipedia Page https://de.wikipedia.org/wiki/Barometrische_H%C3%B6henformel
//On the website the formula sets the outside temperature at 15°C and the pressure at sea level at 1013.25hPa as constant values
//For more precise measurements, change this constants to actual values from your location


float BlueDot_BME680::readAltitudeMeter(void)
{
	float heightOutput = 0;
	float tempOutsideKelvin = convertTempKelvin();
	
	heightOutput = readPressure();
	heightOutput = (heightOutput/parameter.pressureSeaLevel);
	heightOutput = pow(heightOutput, 0.190284);
	heightOutput = 1 - heightOutput;	
	heightOutput = heightOutput * tempOutsideKelvin;
	heightOutput = heightOutput / 0.0065;
	return heightOutput;		
}
//##########################################################################
//This is the same formula from before, but it converts the altitude from meters to feet before returning the results


float BlueDot_BME680::readAltitudeFeet(void)
{	
	float heightOutput = 0;
	float tempOutsideKelvin = convertTempKelvin();
	
	heightOutput = readPressure();
	heightOutput = (heightOutput/parameter.pressureSeaLevel);
	heightOutput = pow(heightOutput, 0.190284);
	heightOutput = 1 - heightOutput;
	heightOutput = heightOutput * tempOutsideKelvin;
	heightOutput = heightOutput / 0.0065;
	heightOutput = heightOutput / 0.3048;
	return heightOutput;	
}


//##########################################################################
//DATA READOUT FUNCTIONS - GAS SENSOR
//##########################################################################
//Reading the gas sensor involves four steps
//This is the first step
//First we write the target temperature for the hot plate on the Arduino Sketch
//This is defined as parameter.target_temp
//This function uses the target_temp value to calculate the resistance of the hot plate
//The BME680 will use this resistance to heat up the gas sensor hot plate
//According to the datasheet the sensor needs a target temperature between 200°C and 400°C to work properly


void BlueDot_BME680::calculateHotPlateRes(void)
{
	double var1;
	double var2;
	double var3;
	double var4;
	double var5;
	
	
	var1 = ((double)bme680_coefficients.par_g1 / 16.0) + 49.0;
	var2 = (((double)bme680_coefficients.par_g2 / 32768.0) * 0.0005) + 0.00235;
	var3 = (double)bme680_coefficients.par_g3 / 1024.0;
	var4 = var1 * (1.0 + (var2 * (double)parameter.target_temp));
	var5 = var4 + (var3 * (double)readTempC());
	
	bme680_coefficients.res_heat_0 = (uint8_t)(3.4 * ((var5 * (4.0 / (4.0 + (double)bme680_coefficients.res_heat_range)) *
										(1.0 / (1.0 + ((double)bme680_coefficients.res_heat_val * 0.002)))) - 25));
	
	writeByte(BME680_RES_HEAT_0, (byte)bme680_coefficients.res_heat_0);	
	
}
//##########################################################################
//This is the second step
//Here we define how long the hot plate should keep the target temperature
//Here I chose a fixed duration of 148 ms for all measurements
//If you wish to change the duration, please refer to the datasheet (page 30, chapter 5.3.3.3)


void BlueDot_BME680::calculateHotPlateTime(void)
{
	int8_t heat_time_0 = 0b01100101;					//148 ms heating duration
	writeByte(BME680_GAS_WAIT_0, heat_time_0);	
	
}
//##########################################################################
//This is the third step
//Now we are sending the heating duration and the target resistance (calculated from the target temperature) to the sensor
//This is done by writing to the register BME680_CTRL_GAS_1 (register 071h)
//Finally by writing the bit 4 of this register to logic 1 we enable the gas sensor measurement


void BlueDot_BME680::setHotPlateProfile(void)
{
	byte value;
	value = (((byte)parameter.hotplate_profile) & 0b00001111) | 0b00010000;
	writeByte(BME680_CTRL_GAS_1, value);
	
	
}
//##########################################################################
bool BlueDot_BME680::readStatus(void)
{
	uint8_t newDataBit = ((readByte(0x1D) & 0b10000000) >> 7);      
	
  if (newDataBit == 1) 
  {                                                  
    uint8_t gasValidBit = ((readByte(0x2B) & 0b00100000) >> 5);
    uint8_t heaterStabilityBit = ((readByte(0x2B) & 0b00010000) >> 4);

    if ((gasValidBit == 1) & (heaterStabilityBit == 1))
    {
		
		return 1;
    }

    else
    {
      return 0;
    }
    
  }

  else
  {
    return 0;
  }

}

//##########################################################################
float BlueDot_BME680::readGas(void)
//This is the last and final step
//Here we read the resistance of the gas sensor
//This is done on the loop function

{
	
	for (int8_t i = 0; i = 10; i++)
	{
		if (readStatus() == 1)
		{
			int16_t gas_r;	
			gas_r = (uint16_t)readByte(BME680_GAS_MSB) << 2;
			gas_r |= (uint16_t)((uint16_t)readByte(BME680_GAS_LSB) >> 6);	
	
			int8_t gas_range;
			gas_range = (uint8_t)readByte(BME680_GAS_LSB) & 0b00001111;
	
			double var1;
			float gas_res;
	
			var1 = (1340.0 + 5.0 * bme680_coefficients.range_switching_error) * const_array1[gas_range];
			gas_res = var1 * const_array2[gas_range] / (gas_r - 512.0 + var1);	

			return gas_res;
		}
		
		delay(20);
	}
	
	return 0;	
	
}

//##########################################################################
//BASIC FUNCTIONS
//##########################################################################
void BlueDot_BME680::writeByte(byte reg, byte value)
{	
	
	Wire.beginTransmission(parameter.I2CAddress);
	Wire.write(reg);
	Wire.write(value);
	Wire.endTransmission();	
}
//##########################################################################
uint8_t BlueDot_BME680::readByte(byte reg)
{
	uint8_t value;	
	
	Wire.beginTransmission(parameter.I2CAddress);
	Wire.write(reg);
	Wire.endTransmission();
	Wire.requestFrom(parameter.I2CAddress,1);		
	value = Wire.read();		
	return value;	
}



