/*
 * Copyright (C) 2016 Southern Storm Software, Pty Ltd.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included
 * in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
 * OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
This example runs tests on the SHAKE128 implementation to verify
correct behaviour.
*/

#include <Crypto.h>
#include <SHAKE.h>
#include <string.h>
#if defined(__AVR__)
#include <avr/pgmspace.h>
#else
#define PROGMEM
#define memcpy_P(d, s, l)   memcpy((d), (s), (l))
#endif

#define MAX_HASH_DATA_SIZE  167
#define MAX_SHAKE_OUTPUT    256

struct TestHashVectorSHAKE
{
    const char *name;
    uint8_t data[MAX_HASH_DATA_SIZE];
    size_t dataLen;
    uint8_t hash[MAX_SHAKE_OUTPUT];
};

// Some test vectors from https://github.com/gvanas/KeccakCodePackage
static TestHashVectorSHAKE const testVectorSHAKE128_1 PROGMEM = {
    "SHAKE128 #1",
    {0},
    0,
    {0x7F, 0x9C, 0x2B, 0xA4, 0xE8, 0x8F, 0x82, 0x7D,
     0x61, 0x60, 0x45, 0x50, 0x76, 0x05, 0x85, 0x3E,
     0xD7, 0x3B, 0x80, 0x93, 0xF6, 0xEF, 0xBC, 0x88,
     0xEB, 0x1A, 0x6E, 0xAC, 0xFA, 0x66, 0xEF, 0x26,
     0x3C, 0xB1, 0xEE, 0xA9, 0x88, 0x00, 0x4B, 0x93,
     0x10, 0x3C, 0xFB, 0x0A, 0xEE, 0xFD, 0x2A, 0x68,
     0x6E, 0x01, 0xFA, 0x4A, 0x58, 0xE8, 0xA3, 0x63,
     0x9C, 0xA8, 0xA1, 0xE3, 0xF9, 0xAE, 0x57, 0xE2,
     0x35, 0xB8, 0xCC, 0x87, 0x3C, 0x23, 0xDC, 0x62,
     0xB8, 0xD2, 0x60, 0x16, 0x9A, 0xFA, 0x2F, 0x75,
     0xAB, 0x91, 0x6A, 0x58, 0xD9, 0x74, 0x91, 0x88,
     0x35, 0xD2, 0x5E, 0x6A, 0x43, 0x50, 0x85, 0xB2,
     0xBA, 0xDF, 0xD6, 0xDF, 0xAA, 0xC3, 0x59, 0xA5,
     0xEF, 0xBB, 0x7B, 0xCC, 0x4B, 0x59, 0xD5, 0x38,
     0xDF, 0x9A, 0x04, 0x30, 0x2E, 0x10, 0xC8, 0xBC,
     0x1C, 0xBF, 0x1A, 0x0B, 0x3A, 0x51, 0x20, 0xEA,
     0x17, 0xCD, 0xA7, 0xCF, 0xAD, 0x76, 0x5F, 0x56,
     0x23, 0x47, 0x4D, 0x36, 0x8C, 0xCC, 0xA8, 0xAF,
     0x00, 0x07, 0xCD, 0x9F, 0x5E, 0x4C, 0x84, 0x9F,
     0x16, 0x7A, 0x58, 0x0B, 0x14, 0xAA, 0xBD, 0xEF,
     0xAE, 0xE7, 0xEE, 0xF4, 0x7C, 0xB0, 0xFC, 0xA9,
     0x76, 0x7B, 0xE1, 0xFD, 0xA6, 0x94, 0x19, 0xDF,
     0xB9, 0x27, 0xE9, 0xDF, 0x07, 0x34, 0x8B, 0x19,
     0x66, 0x91, 0xAB, 0xAE, 0xB5, 0x80, 0xB3, 0x2D,
     0xEF, 0x58, 0x53, 0x8B, 0x8D, 0x23, 0xF8, 0x77,
     0x32, 0xEA, 0x63, 0xB0, 0x2B, 0x4F, 0xA0, 0xF4,
     0x87, 0x33, 0x60, 0xE2, 0x84, 0x19, 0x28, 0xCD,
     0x60, 0xDD, 0x4C, 0xEE, 0x8C, 0xC0, 0xD4, 0xC9,
     0x22, 0xA9, 0x61, 0x88, 0xD0, 0x32, 0x67, 0x5C,
     0x8A, 0xC8, 0x50, 0x93, 0x3C, 0x7A, 0xFF, 0x15,
     0x33, 0xB9, 0x4C, 0x83, 0x4A, 0xDB, 0xB6, 0x9C,
     0x61, 0x15, 0xBA, 0xD4, 0x69, 0x2D, 0x86, 0x19}
};
static TestHashVectorSHAKE const testVectorSHAKE128_2 PROGMEM = {
    "SHAKE128 #2",
    {0x1F, 0x87, 0x7C},
    3,
    {0xE2, 0xD3, 0x14, 0x46, 0x69, 0xAB, 0x57, 0x83,
     0x47, 0xFC, 0xCA, 0x0B, 0x57, 0x27, 0x83, 0xA2,
     0x69, 0xA8, 0xCF, 0x9A, 0xDD, 0xA4, 0xD8, 0x77,
     0x82, 0x05, 0x3D, 0x80, 0xD5, 0xF0, 0xFD, 0xD2,
     0x78, 0x35, 0xCF, 0x88, 0x30, 0x36, 0xE5, 0x36,
     0xCE, 0x76, 0xFE, 0xF6, 0x89, 0xA5, 0xE7, 0xBD,
     0x64, 0x6A, 0x7F, 0xB7, 0xD7, 0x4F, 0x09, 0x01,
     0x93, 0xB2, 0x39, 0x0E, 0x61, 0x47, 0x59, 0xB7,
     0xEB, 0x7D, 0xE9, 0x15, 0xA3, 0x83, 0x28, 0x74,
     0x58, 0x90, 0xB1, 0xEF, 0x1E, 0x7A, 0xED, 0x78,
     0x16, 0x8E, 0x99, 0x6D, 0x7A, 0xC7, 0x74, 0xD4,
     0x7F, 0x8F, 0x11, 0x8B, 0x3E, 0x00, 0xA7, 0xBD,
     0x15, 0x11, 0x31, 0xBA, 0x37, 0x05, 0xAE, 0x81,
     0xB5, 0x7F, 0xB7, 0xCB, 0xFF, 0xE1, 0x14, 0xE2,
     0xF4, 0xC3, 0xCA, 0x15, 0x2B, 0x88, 0x74, 0xFB,
     0x90, 0x6E, 0x86, 0x28, 0x40, 0x62, 0x4E, 0x02,
     0xBB, 0xF9, 0x50, 0x2E, 0x46, 0xD8, 0x88, 0x84,
     0x33, 0xA3, 0x8E, 0x82, 0xE0, 0x4C, 0xAA, 0xCB,
     0x60, 0x01, 0x92, 0x22, 0xD4, 0x33, 0xE8, 0xF2,
     0xE7, 0x58, 0xBD, 0x41, 0xAA, 0xB3, 0x95, 0xBF,
     0x83, 0x61, 0x1F, 0xD0, 0xC3, 0xF7, 0xFD, 0x51,
     0x73, 0x30, 0x61, 0x82, 0x44, 0x9B, 0x9A, 0x22,
     0xC4, 0x01, 0x3F, 0x22, 0x63, 0xB4, 0x1E, 0xAC,
     0x4D, 0x0E, 0xDA, 0x16, 0x85, 0x49, 0x61, 0xFB,
     0xAA, 0x6A, 0xD0, 0x4A, 0x89, 0xE7, 0x2A, 0x60,
     0x2A, 0xC5, 0x96, 0x59, 0xEC, 0x2A, 0x60, 0xC1,
     0xD0, 0x20, 0xBA, 0xCC, 0x74, 0xA7, 0x11, 0xD4,
     0x25, 0x4A, 0x2E, 0xCC, 0x5F, 0x8F, 0x06, 0x27,
     0xB4, 0xF7, 0x2A, 0xE1, 0x30, 0xC5, 0x05, 0x90,
     0xF8, 0xB9, 0x1C, 0x52, 0x95, 0x7B, 0x79, 0x5D,
     0x12, 0xDA, 0x09, 0xBD, 0xD4, 0x0D, 0x41, 0xE3,
     0xCD, 0x48, 0xE3, 0x0E, 0x37, 0xFE, 0x5F, 0xD0}
};
static TestHashVectorSHAKE const testVectorSHAKE128_3 PROGMEM = {
    "SHAKE128 #3",
    {0x0D, 0x8D, 0x09, 0xAE, 0xD1, 0x9F, 0x10, 0x13,
     0x96, 0x9C, 0xE5, 0xE7, 0xEB, 0x92, 0xF8, 0x3A,
     0x20, 0x9A, 0xE7, 0x6B, 0xE3, 0x1C, 0x75, 0x48,
     0x44, 0xEA, 0x91, 0x16, 0xCE, 0xB3, 0x9A, 0x22,
     0xEB, 0xB6, 0x00, 0x30, 0x17, 0xBB, 0xCF, 0x26,
     0x55, 0x5F, 0xA6, 0x62, 0x41, 0x85, 0x18, 0x7D,
     0xB8, 0xF0, 0xCB, 0x35, 0x64, 0xB8, 0xB1, 0xC0,
     0x6B, 0xF6, 0x85, 0xD4, 0x7F, 0x32, 0x86, 0xED,
     0xA2, 0x0B, 0x83, 0x35, 0x8F, 0x59, 0x9D, 0x20,
     0x44, 0xBB, 0xF0, 0x58, 0x3F, 0xAB, 0x8D, 0x78,
     0xF8, 0x54, 0xFE, 0x0A, 0x59, 0x61, 0x83, 0x23,
     0x0C, 0x5E, 0xF8, 0xE5, 0x44, 0x26, 0x75, 0x0E,
     0xAF, 0x2C, 0xC4, 0xE2, 0x9D, 0x3B, 0xDD, 0x03,
     0x7E, 0x73, 0x4D, 0x86, 0x3C, 0x2B, 0xD9, 0x78,
     0x9B, 0x4C, 0x24, 0x30, 0x96, 0x13, 0x8F, 0x76,
     0x72, 0xC2, 0x32, 0x31, 0x4E, 0xFF, 0xDF, 0xC6,
     0x51, 0x34, 0x27, 0xE2, 0xDA, 0x76, 0x91, 0x6B,
     0x52, 0x48, 0x93, 0x3B, 0xE3, 0x12, 0xEB, 0x5D,
     0xDE, 0x4C, 0xF7, 0x08, 0x04, 0xFB, 0x25, 0x8A,
     0xC5, 0xFB, 0x82, 0xD5, 0x8D, 0x08, 0x17, 0x7A,
     0xC6, 0xF4, 0x75, 0x60, 0x17, 0xFF, 0xF5},
    167,
    {0xC7, 0x3D, 0x8F, 0xAA, 0xB5, 0xD0, 0xB4, 0xD6,
     0x60, 0xBD, 0x50, 0x82, 0xE4, 0x4C, 0x3C, 0xAC,
     0x97, 0xE6, 0x16, 0x48, 0xBE, 0x0A, 0x04, 0xB1,
     0x16, 0x72, 0x4E, 0x6F, 0x6B, 0x65, 0x76, 0x84,
     0x67, 0x4B, 0x4B, 0x0E, 0x90, 0xD0, 0xAE, 0x96,
     0xC0, 0x85, 0x3E, 0xBD, 0x83, 0x7B, 0xD8, 0x24,
     0x9A, 0xDB, 0xD3, 0xB6, 0x0A, 0x1A, 0xD1, 0xFC,
     0xF8, 0xA6, 0xAB, 0x8E, 0x2F, 0x5A, 0xA7, 0xFF,
     0x19, 0x7A, 0x3D, 0x7D, 0xBE, 0xDE, 0xFB, 0x43,
     0x3B, 0x61, 0x35, 0x36, 0xAE, 0xC4, 0xD6, 0x55,
     0xB7, 0xBC, 0xD7, 0x78, 0x52, 0x6B, 0xE6, 0x67,
     0x84, 0x7A, 0xCD, 0x2E, 0x05, 0x64, 0xD9, 0x6C,
     0xE5, 0x14, 0x0C, 0x91, 0x35, 0x7F, 0xAD, 0xE0,
     0x00, 0xEF, 0xCB, 0x40, 0x45, 0x7E, 0x1B, 0x6C,
     0xED, 0x41, 0xFA, 0x10, 0x2E, 0x36, 0xE7, 0x99,
     0x79, 0x2D, 0xB0, 0x3E, 0x9A, 0x40, 0xC7, 0x99,
     0xBC, 0xA9, 0x12, 0x62, 0x94, 0x8E, 0x17, 0x60,
     0x50, 0x65, 0xFB, 0xF6, 0x38, 0xFB, 0x40, 0xA1,
     0x57, 0xB4, 0x5C, 0xF7, 0x91, 0x1A, 0x75, 0x3D,
     0x0D, 0x20, 0x5D, 0xF8, 0x47, 0x16, 0xA5, 0x71,
     0x12, 0xBE, 0xAB, 0x44, 0xF6, 0x20, 0x1F, 0xF7,
     0x5A, 0xAD, 0xE0, 0xBA, 0xFB, 0xA5, 0x04, 0x74,
     0x5C, 0xFE, 0x23, 0xE4, 0xE6, 0x0E, 0x67, 0xE3,
     0x99, 0x36, 0x22, 0xAE, 0xD7, 0x3A, 0x1D, 0xD6,
     0xA4, 0x65, 0xBD, 0x45, 0x3D, 0xD3, 0xC5, 0xBA,
     0x7D, 0x2C, 0xDF, 0x3F, 0x1D, 0x39, 0x37, 0x6A,
     0x67, 0xC2, 0x3E, 0x55, 0x5F, 0x5A, 0xCF, 0x25,
     0xBC, 0xE1, 0xE5, 0x5F, 0x30, 0x72, 0x52, 0xB9,
     0xAA, 0xC2, 0xC0, 0xA3, 0x9C, 0x88, 0x5C, 0x7E,
     0x44, 0xF2, 0x04, 0xCB, 0x82, 0x1C, 0x0D, 0x37,
     0xA2, 0x2D, 0xE3, 0xA7, 0x1F, 0x3A, 0x19, 0x09,
     0xB1, 0x1B, 0x71, 0x81, 0xC4, 0x2B, 0xE9, 0xB7}
};

SHAKE128 shake128;
TestHashVectorSHAKE tst;
uint8_t output[MAX_SHAKE_OUTPUT];

bool testSHAKE_N(SHAKE *shake, const struct TestHashVectorSHAKE *test, size_t inc, bool printName = false)
{
    size_t size;
    size_t posn, len;

    // Copy the test case out of program memory.
    memcpy_P(&tst, test, sizeof(tst));
    test = &tst;

    // Print the test name if necessary.
    if (printName) {
        Serial.print(test->name);
        Serial.print(" ... ");
    }

    // Hash the input data.
    if (!inc)
        inc = 1;
    size = test->dataLen;
    shake->reset();
    for (posn = 0; posn < size; posn += inc) {
        len = size - posn;
        if (len > inc)
            len = inc;
        shake->update(test->data + posn, len);
    }

    // Generate data using extend() and check it.
    size = MAX_SHAKE_OUTPUT;
    for (posn = 0; posn < size; posn += inc) {
        len = size - posn;
        if (len > inc)
            len = inc;
        shake->extend(output + posn, len);
    }
    if (memcmp(output, test->hash, size) != 0)
        return false;

    // Restart and use encrypt() this time.
    shake->reset();
    shake->update(test->data, test->dataLen);
    for (posn = 0; posn < size; ++posn)
        output[posn] = (uint8_t)(posn + 0xAA);
    for (posn = 0; posn < size; posn += inc) {
        len = size - posn;
        if (len > inc)
            len = inc;
        shake->encrypt(output + posn, output + posn, len);
    }
    for (posn = 0; posn < size; ++posn)
        output[posn] ^= (uint8_t)(posn + 0xAA);
    if (memcmp(output, test->hash, size) != 0)
        return false;

    return true;
}

void testSHAKE(SHAKE *shake, const struct TestHashVectorSHAKE *test)
{
    bool ok;
    size_t dataLen;

    memcpy_P(&dataLen, &(test->dataLen), sizeof(size_t));

    ok  = testSHAKE_N(shake, test, dataLen, true);
    ok &= testSHAKE_N(shake, test, 1);
    ok &= testSHAKE_N(shake, test, 2);
    ok &= testSHAKE_N(shake, test, 5);
    ok &= testSHAKE_N(shake, test, 8);
    ok &= testSHAKE_N(shake, test, 13);
    ok &= testSHAKE_N(shake, test, 16);
    ok &= testSHAKE_N(shake, test, 24);
    ok &= testSHAKE_N(shake, test, 63);
    ok &= testSHAKE_N(shake, test, 64);

    if (ok)
        Serial.println("Passed");
    else
        Serial.println("Failed");
}

void perfUpdate(SHAKE *shake)
{
    unsigned long start;
    unsigned long elapsed;
    int count;

    Serial.print("Updating ... ");

    for (size_t posn = 0; posn < MAX_SHAKE_OUTPUT; ++posn)
        output[posn] = (uint8_t)posn;

    shake->reset();
    start = micros();
    for (count = 0; count < 300; ++count) {
        shake->update(output, MAX_SHAKE_OUTPUT);
    }
    shake->extend(output, 0);   // Force a finalize after the update.
    elapsed = micros() - start;

    Serial.print(elapsed / (MAX_SHAKE_OUTPUT * 300.0));
    Serial.print("us per byte, ");
    Serial.print((MAX_SHAKE_OUTPUT * 300.0 * 1000000.0) / elapsed);
    Serial.println(" bytes per second");
}

void perfExtend(SHAKE *shake)
{
    unsigned long start;
    unsigned long elapsed;
    int count;

    Serial.print("Extending ... ");

    for (size_t posn = 0; posn < MAX_SHAKE_OUTPUT; ++posn)
        output[posn] = (uint8_t)posn;

    shake->reset();
    shake->update(output, MAX_SHAKE_OUTPUT);
    shake->extend(output, 0);   // Force a finalize after the update.
    start = micros();
    for (count = 0; count < 300; ++count) {
        shake->extend(output, MAX_SHAKE_OUTPUT);
    }
    elapsed = micros() - start;

    Serial.print(elapsed / (MAX_SHAKE_OUTPUT * 300.0));
    Serial.print("us per byte, ");
    Serial.print((MAX_SHAKE_OUTPUT * 300.0 * 1000000.0) / elapsed);
    Serial.println(" bytes per second");
}

void perfEncrypt(SHAKE *shake)
{
    unsigned long start;
    unsigned long elapsed;
    int count;

    Serial.print("Encrypting ... ");

    for (size_t posn = 0; posn < MAX_SHAKE_OUTPUT; ++posn)
        output[posn] = (uint8_t)posn;

    shake->reset();
    shake->update(output, MAX_SHAKE_OUTPUT);
    shake->extend(output, 0);   // Force a finalize after the update.
    start = micros();
    for (count = 0; count < 300; ++count) {
        shake->encrypt(output, output, MAX_SHAKE_OUTPUT);
    }
    elapsed = micros() - start;

    Serial.print(elapsed / (MAX_SHAKE_OUTPUT * 300.0));
    Serial.print("us per byte, ");
    Serial.print((MAX_SHAKE_OUTPUT * 300.0 * 1000000.0) / elapsed);
    Serial.println(" bytes per second");
}

void setup()
{
    Serial.begin(9600);

    Serial.println();

    Serial.print("State Size ...");
    Serial.println(sizeof(SHAKE128));
    Serial.println();

    Serial.println("Test Vectors:");
    testSHAKE(&shake128, &testVectorSHAKE128_1);
    testSHAKE(&shake128, &testVectorSHAKE128_2);
    testSHAKE(&shake128, &testVectorSHAKE128_3);

    Serial.println();

    Serial.println("Performance Tests:");
    perfUpdate(&shake128);
    perfExtend(&shake128);
    perfEncrypt(&shake128);
}

void loop()
{
}
