// Copyright 2017 David Conran

#include "IRsend.h"
#include "IRsend_test.h"
#include "gtest/gtest.h"

// Tests for sendGree().

// Test sending typical data only.
TEST(TestSendGreeChars, SendData) {
  IRsendTest irsend(4);
  irsend.begin();

  uint8_t gree_code[GREE_STATE_LENGTH] = {
      0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF};
  irsend.reset();
  irsend.sendGree(gree_code);
  EXPECT_EQ(
      "m9000s4000"
      "m620s540m620s1600m620s540m620s540m620s1600m620s540m620s540m620s540"
      "m620s540m620s540m620s1600m620s540m620s1600m620s1600m620s540m620s540"
      "m620s540m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540"
      "m620s540m620s540m620s540m620s1600m620s1600m620s1600m620s1600m620s540"
      "m620s540m620s1600m620s540"
      "m620s19000"
      "m620s540m620s540m620s540m620s540m620s1600m620s540m620s540m620s1600"
      "m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540m620s1600"
      "m620s1600m620s540m620s1600m620s1600m620s540m620s540m620s1600m620s1600"
      "m620s1600m620s1600m620s1600m620s1600m620s540m620s1600m620s1600m620s1600"
      "m620s19000", irsend.outputStr());
}

TEST(TestSendGreeUint64, SendData) {
  IRsendTest irsend(4);
  irsend.begin();

  irsend.reset();
  irsend.sendGree(0x1234567890ABCDEF);
  EXPECT_EQ(
      "m9000s4000"
      "m620s540m620s1600m620s540m620s540m620s1600m620s540m620s540m620s540"
      "m620s540m620s540m620s1600m620s540m620s1600m620s1600m620s540m620s540"
      "m620s540m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540"
      "m620s540m620s540m620s540m620s1600m620s1600m620s1600m620s1600m620s540"
      "m620s540m620s1600m620s540"
      "m620s19000"
      "m620s540m620s540m620s540m620s540m620s1600m620s540m620s540m620s1600"
      "m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540m620s1600"
      "m620s1600m620s540m620s1600m620s1600m620s540m620s540m620s1600m620s1600"
      "m620s1600m620s1600m620s1600m620s1600m620s540m620s1600m620s1600m620s1600"
      "m620s19000", irsend.outputStr());
}

// Test sending with repeats.
TEST(TestSendGreeChars, SendWithRepeats) {
  IRsendTest irsend(4);
  irsend.begin();

  irsend.reset();
  uint8_t gree_code[GREE_STATE_LENGTH] = {
      0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF};
  irsend.reset();

  irsend.sendGree(gree_code, GREE_STATE_LENGTH, 1);
  EXPECT_EQ(
    "m9000s4000"
    "m620s540m620s1600m620s540m620s540m620s1600m620s540m620s540m620s540"
    "m620s540m620s540m620s1600m620s540m620s1600m620s1600m620s540m620s540"
    "m620s540m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540"
    "m620s540m620s540m620s540m620s1600m620s1600m620s1600m620s1600m620s540"
    "m620s540m620s1600m620s540"
    "m620s19000"
    "m620s540m620s540m620s540m620s540m620s1600m620s540m620s540m620s1600"
    "m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540m620s1600"
    "m620s1600m620s540m620s1600m620s1600m620s540m620s540m620s1600m620s1600"
    "m620s1600m620s1600m620s1600m620s1600m620s540m620s1600m620s1600m620s1600"
    "m620s19000"
    "m9000s4000"
    "m620s540m620s1600m620s540m620s540m620s1600m620s540m620s540m620s540"
    "m620s540m620s540m620s1600m620s540m620s1600m620s1600m620s540m620s540"
    "m620s540m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540"
    "m620s540m620s540m620s540m620s1600m620s1600m620s1600m620s1600m620s540"
    "m620s540m620s1600m620s540"
    "m620s19000"
    "m620s540m620s540m620s540m620s540m620s1600m620s540m620s540m620s1600"
    "m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540m620s1600"
    "m620s1600m620s540m620s1600m620s1600m620s540m620s540m620s1600m620s1600"
    "m620s1600m620s1600m620s1600m620s1600m620s540m620s1600m620s1600m620s1600"
    "m620s19000", irsend.outputStr());
}

TEST(TestSendGreeUint64, SendWithRepeats) {
  IRsendTest irsend(4);
  irsend.begin();

  irsend.reset();
  irsend.sendGree(0x1234567890ABCDEF, GREE_BITS, 1);
  EXPECT_EQ(
    "m9000s4000"
    "m620s540m620s1600m620s540m620s540m620s1600m620s540m620s540m620s540"
    "m620s540m620s540m620s1600m620s540m620s1600m620s1600m620s540m620s540"
    "m620s540m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540"
    "m620s540m620s540m620s540m620s1600m620s1600m620s1600m620s1600m620s540"
    "m620s540m620s1600m620s540"
    "m620s19000"
    "m620s540m620s540m620s540m620s540m620s1600m620s540m620s540m620s1600"
    "m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540m620s1600"
    "m620s1600m620s540m620s1600m620s1600m620s540m620s540m620s1600m620s1600"
    "m620s1600m620s1600m620s1600m620s1600m620s540m620s1600m620s1600m620s1600"
    "m620s19000"
    "m9000s4000"
    "m620s540m620s1600m620s540m620s540m620s1600m620s540m620s540m620s540"
    "m620s540m620s540m620s1600m620s540m620s1600m620s1600m620s540m620s540"
    "m620s540m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540"
    "m620s540m620s540m620s540m620s1600m620s1600m620s1600m620s1600m620s540"
    "m620s540m620s1600m620s540"
    "m620s19000"
    "m620s540m620s540m620s540m620s540m620s1600m620s540m620s540m620s1600"
    "m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540m620s1600"
    "m620s1600m620s540m620s1600m620s1600m620s540m620s540m620s1600m620s1600"
    "m620s1600m620s1600m620s1600m620s1600m620s540m620s1600m620s1600m620s1600"
    "m620s19000", irsend.outputStr());
}

// Test sending atypical sizes.
TEST(TestSendGreeChars, SendUnexpectedSizes) {
  IRsendTest irsend(4);
  irsend.begin();

  uint8_t gree_short_code[GREE_STATE_LENGTH - 1] = {
      0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD};
  uint8_t gree_long_code[GREE_STATE_LENGTH + 1] = {
      0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF, 0x12};
  irsend.reset();
  irsend.sendGree(gree_short_code, GREE_STATE_LENGTH - 1);
  ASSERT_EQ("", irsend.outputStr());

  irsend.reset();
  irsend.sendGree(gree_long_code, GREE_STATE_LENGTH + 1);
  ASSERT_EQ(
    "m9000s4000"
    "m620s540m620s1600m620s540m620s540m620s1600m620s540m620s540m620s540"
    "m620s540m620s540m620s1600m620s540m620s1600m620s1600m620s540m620s540"
    "m620s540m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540"
    "m620s540m620s540m620s540m620s1600m620s1600m620s1600m620s1600m620s540"
    "m620s540m620s1600m620s540"
    "m620s19000"
    "m620s540m620s540m620s540m620s540m620s1600m620s540m620s540m620s1600"
    "m620s1600m620s1600m620s540m620s1600m620s540m620s1600m620s540m620s1600"
    "m620s1600m620s540m620s1600m620s1600m620s540m620s540m620s1600m620s1600"
    "m620s1600m620s1600m620s1600m620s1600m620s540m620s1600m620s1600m620s1600"
    "m620s540m620s1600m620s540m620s540m620s1600m620s540m620s540m620s540"
    "m620s19000", irsend.outputStr());
}

TEST(TestSendGreeUint64, SendUnexpectedSizes) {
  IRsendTest irsend(4);
  irsend.begin();

  irsend.reset();
  irsend.sendGree(0x1234567890ABCDEF, GREE_BITS - 1);
  ASSERT_EQ("", irsend.outputStr());

  irsend.reset();
  irsend.sendGree(0x1234567890ABCDEF, GREE_BITS + 1);
  ASSERT_EQ("", irsend.outputStr());
}

TEST(TestSendGree, CompareUint64ToCharResults) {
  IRsendTest irsend_chars(4);
  IRsendTest irsend_uint64(0);

  uint8_t gree_code[GREE_STATE_LENGTH] = {
      0x12, 0x34, 0x56, 0x78, 0x90, 0xAB, 0xCD, 0xEF};

  irsend_chars.begin();
  irsend_uint64.begin();

  irsend_chars.reset();
  irsend_uint64.reset();
  irsend_chars.sendGree(gree_code);
  irsend_uint64.sendGree(0x1234567890ABCDEF);
  ASSERT_EQ(irsend_chars.outputStr(), irsend_uint64.outputStr());

  uint8_t gree_zero_code[GREE_STATE_LENGTH] = {
      0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  irsend_chars.reset();
  irsend_uint64.reset();
  irsend_chars.sendGree(gree_zero_code);
  irsend_uint64.sendGree((uint64_t) 0x0);
  ASSERT_EQ(irsend_chars.outputStr(), irsend_uint64.outputStr());
}
