#!/usr/bin/env python3
"""
Generador de projectes Qt amb suport per Desktop, WebAssembly i Android
Autor: Jordi Binefa
Data: 2025-12-09
Versió: 2.0 - Adaptat per Qt 6.9.1
"""

import os
import sys
from pathlib import Path

def create_project():
    print("=" * 60)
    print("  GENERADOR DE PROJECTES QT MULTIPLATAFORMA")
    print("=" * 60)
    
    # Demanar informació del projecte
    project_name = input("\n📦 Nom del projecte (ex: mqtt00): ").strip()
    if not project_name:
        print("❌ El nom del projecte no pot estar buit!")
        sys.exit(1)
    
    app_name = input(f"📱 Nom de l'aplicació [app{project_name}]: ").strip()
    if not app_name:
        app_name = f"app{project_name}"
    
    project_version = input("📌 Versió del projecte [0.1]: ").strip() or "0.1"
    
    qml_uri = input(f"🎨 URI QML [premeu Enter per deixar buit o ex: com.example.{project_name}]: ").strip()
    
    # Components Qt (fixos)
    qt_components = "Core Quick Network Mqtt WebSockets"
    
    # Tipus de projecte (sempre 3)
    project_type = "3"
    
    # Crear estructura de directoris
    project_dir = Path(project_name)
    if project_dir.exists():
        overwrite = input(f"\n⚠️  El directori '{project_name}' ja existeix. Sobreescriure? [s/N]: ")
        if overwrite.lower() != 's':
            print("❌ Operació cancel·lada")
            sys.exit(0)
    
    project_dir.mkdir(exist_ok=True)
    
    # Generar CMakeLists.txt
    cmakelists_content = generate_cmakelists(
        project_name=project_name,
        app_name=app_name,
        version=project_version,
        qml_uri=qml_uri,
        components=qt_components,
        project_type=project_type
    )
    
    (project_dir / "CMakeLists.txt").write_text(cmakelists_content, encoding='utf-8')
    
    # Generar main.cpp
    main_cpp_content = generate_main_cpp(app_name, qml_uri)
    (project_dir / "main.cpp").write_text(main_cpp_content, encoding='utf-8')
    
    # Generar main.qml
    main_qml_content = generate_main_qml(project_name)
    (project_dir / "main.qml").write_text(main_qml_content, encoding='utf-8')
    
    # Generar qml.qrc
    qml_qrc_content = """<RCC>
    <qresource prefix="/">
        <file>main.qml</file>
    </qresource>
</RCC>"""
    (project_dir / "qml.qrc").write_text(qml_qrc_content, encoding='utf-8')
    
    # Generar .gitignore
    gitignore_content = """build*/
*.user
*.autosave
.qtc_clangd/
CMakeLists.txt.user*"""
    (project_dir / ".gitignore").write_text(gitignore_content, encoding='utf-8')
    
    print(f"\n✅ Projecte '{project_name}' creat correctament!")
    print(f"📁 Ubicació: {project_dir.absolute()}")
    print(f"\n📝 Passos següents:")
    print(f"   1. Copia a la nova carpeta els arxius (.h i .cpp): connexio, mqttbasic i websocketiodevice")
    print(f"   2. Obre Qt Creator")
    print(f"   3. File → Open File or Project...")
    print(f"   4. Selecciona: {project_dir.absolute() / 'CMakeLists.txt'}")
    print(f"   5. Configura els kits (Desktop, Android, WebAssembly)")
    print(f"   6. Build → Build All")

def generate_cmakelists(project_name, app_name, version, qml_uri, components, project_type):
    # Sempre tipus 3: main.cpp + connexio + mqttbasic + websocketiodevice
    sources_in_qml_module = """connexio.cpp
            mqttbasic.cpp
            websocketiodevice.cpp"""
    
    headers_in_qml_module = """connexio.h
            mqttbasic.h
            websocketiodevice.h"""
    
    return f"""cmake_minimum_required(VERSION 3.16)
project({project_name} VERSION {version} LANGUAGES CXX)

set(CMAKE_AUTOUIC ON)
set(CMAKE_AUTOMOC ON)
set(CMAKE_AUTORCC ON)
set(CMAKE_CXX_STANDARD 17)
set(CMAKE_CXX_STANDARD_REQUIRED ON)

# Opcions específiques per WebAssembly - ABANS de find_package
if(EMSCRIPTEN)
    message(STATUS "Compilant per WebAssembly")
    set(CMAKE_CXX_FLAGS "${{CMAKE_CXX_FLAGS}} -s ALLOW_MEMORY_GROWTH=1")
    set(CMAKE_CXX_FLAGS "${{CMAKE_CXX_FLAGS}} -s ASYNCIFY=1")
    set(CMAKE_CXX_FLAGS "${{CMAKE_CXX_FLAGS}} -s WEBSOCKET_URL=wss://")
    set(CMAKE_CXX_FLAGS "${{CMAKE_CXX_FLAGS}} -s ERROR_ON_UNDEFINED_SYMBOLS=0")
    set(CMAKE_CXX_FLAGS "${{CMAKE_CXX_FLAGS}} -s EXPORT_ALL=1")
    set(CMAKE_CXX_FLAGS "${{CMAKE_CXX_FLAGS}} -s FORCE_FILESYSTEM=1")
    set(CMAKE_CXX_FLAGS "${{CMAKE_CXX_FLAGS}} -s WASM=1")
    set(CMAKE_CXX_FLAGS "${{CMAKE_CXX_FLAGS}} -s STACK_SIZE=5MB")
    set(CMAKE_CXX_FLAGS "${{CMAKE_CXX_FLAGS}} -s DISABLE_EXCEPTION_CATCHING=0")
endif()

find_package(QT NAMES Qt6 Qt5 REQUIRED COMPONENTS {components})
find_package(Qt${{QT_VERSION_MAJOR}} REQUIRED COMPONENTS {components})

# ============================================================
# CONFIGURACIÓ OPENSSL PER ANDROID - DESPRÉS de find_package
# ============================================================
if(ANDROID)
    message(STATUS "================================================")
    message(STATUS "  Configurant OpenSSL per Android")
    message(STATUS "================================================")
    message(STATUS "  Qt Version: ${{QT_VERSION}}")
    message(STATUS "  Qt Major: ${{QT_VERSION_MAJOR}}, Minor: ${{QT_VERSION_MINOR}}")
    message(STATUS "  Arquitectura: ${{ANDROID_ABI}}")
    
    # Ruta multiplataforma (funciona a Linux, macOS i Windows)
    set(ANDROID_OPENSSL_ROOT "$ENV{{HOME}}/Android/android_openssl")
    
    # Per Qt 6.5+ usar OpenSSL 3.x
    set(OPENSSL_ROOT "${{ANDROID_OPENSSL_ROOT}}/ssl_3/${{ANDROID_ABI}}")
    set(LIB_SSL "libssl_3.so")
    set(LIB_CRYPTO "libcrypto_3.so")
    
    message(STATUS "  Buscant OpenSSL 3.x a: ${{OPENSSL_ROOT}}")
    
    # Verificar que existeixen les llibreries
    if(EXISTS "${{OPENSSL_ROOT}}/${{LIB_SSL}}")
        message(STATUS "  ✓ Trobat: ${{OPENSSL_ROOT}}/${{LIB_SSL}}")
    else()
        message(FATAL_ERROR "  ✗ NO trobat: ${{OPENSSL_ROOT}}/${{LIB_SSL}}")
    endif()
    
    if(EXISTS "${{OPENSSL_ROOT}}/${{LIB_CRYPTO}}")
        message(STATUS "  ✓ Trobat: ${{OPENSSL_ROOT}}/${{LIB_CRYPTO}}")
    else()
        message(FATAL_ERROR "  ✗ NO trobat: ${{OPENSSL_ROOT}}/${{LIB_CRYPTO}}")
    endif()
    
    # CRÍTICO: Configurar les llibreries extra per Android
    set(ANDROID_EXTRA_LIBS
        "${{OPENSSL_ROOT}}/${{LIB_SSL}}"
        "${{OPENSSL_ROOT}}/${{LIB_CRYPTO}}"
        CACHE INTERNAL "Extra libs for Android"
    )
    
    # CRÍTICO: Passar a Qt amb FORCE
    set(QT_ANDROID_EXTRA_LIBS 
        "${{OPENSSL_ROOT}}/${{LIB_SSL}};${{OPENSSL_ROOT}}/${{LIB_CRYPTO}}" 
        CACHE STRING "" FORCE
    )
    
    message(STATUS "  Llibreries configurades:")
    message(STATUS "    - ${{OPENSSL_ROOT}}/${{LIB_SSL}}")
    message(STATUS "    - ${{OPENSSL_ROOT}}/${{LIB_CRYPTO}}")
    message(STATUS "  ✓ OpenSSL configurat correctament per Android")
    message(STATUS "================================================")
endif()

if(${{QT_VERSION_MAJOR}} GREATER_EQUAL 6)
    qt_add_executable({app_name}
        MANUAL_FINALIZATION
        main.cpp
    )

    qt_add_qml_module({app_name}
        URI {qml_uri if qml_uri else project_name}
        VERSION 1.0
        QML_FILES main.qml
        SOURCES
            {sources_in_qml_module}
        {headers_in_qml_module}
        RESOURCES qml.qrc
    )
else()
    qt_add_executable({app_name}
        main.cpp
    )

    qt_add_qml_module({app_name}
        URI {qml_uri if qml_uri else project_name}
        VERSION 1.0
        QML_FILES main.qml
        SOURCES
            {sources_in_qml_module}
        {headers_in_qml_module}
        RESOURCES qml.qrc
    )
endif()

if(${{QT_VERSION}} VERSION_LESS 6.1.0)
    set(BUNDLE_ID_OPTION MACOSX_BUNDLE_GUI_IDENTIFIER com.example.{app_name})
endif()

set_target_properties({app_name} PROPERTIES
    ${{BUNDLE_ID_OPTION}}
    MACOSX_BUNDLE_BUNDLE_VERSION ${{PROJECT_VERSION}}
    MACOSX_BUNDLE_SHORT_VERSION_STRING ${{PROJECT_VERSION_MAJOR}}.${{PROJECT_VERSION_MINOR}}
    MACOSX_BUNDLE TRUE
    WIN32_EXECUTABLE TRUE
)

target_link_libraries({app_name}
    PRIVATE
        {' '.join([f'Qt${{QT_VERSION_MAJOR}}::{comp}' for comp in components.split()])}
)

# ============================================================
# AFEGIR LLIBRERIES OPENSSL AL TARGET (DESPRÉS de target_link_libraries)
# ============================================================
if(ANDROID)
    set_property(TARGET {app_name} 
        APPEND PROPERTY QT_ANDROID_EXTRA_LIBS
        "${{OPENSSL_ROOT}}/${{LIB_SSL}}"
        "${{OPENSSL_ROOT}}/${{LIB_CRYPTO}}"
    )
    
    message(STATUS "================================================")
    message(STATUS "  EXTRA LIBS AFEGIDES AL TARGET {app_name}")
    message(STATUS "================================================")
endif()

include(GNUInstallDirs)
install(TARGETS {app_name}
    BUNDLE DESTINATION .
    LIBRARY DESTINATION ${{CMAKE_INSTALL_LIBDIR}}
    RUNTIME DESTINATION ${{CMAKE_INSTALL_BINDIR}}
)

if(QT_VERSION_MAJOR EQUAL 6)
    qt_finalize_executable({app_name})
endif()
"""

def generate_main_cpp(app_name, qml_uri):
    # Generar URL segons si qml_uri està buit o no
    if not qml_uri:
        url_line = 'const QUrl url(QStringLiteral("qrc:/main.qml"));'
    else:
        # Convertir com.example.mqttClient00 → com/example/mqttClient00
        url_path = qml_uri.replace('.', '/')
        url_line = f'const QUrl url("qrc:/{url_path}/main.qml");'
    
    return f"""#include <QGuiApplication>
#include <QQmlApplicationEngine>

int main(int argc, char *argv[])
{{
    QGuiApplication app(argc, argv);

    QQmlApplicationEngine engine;
    {url_line}
    
    QObject::connect(&engine, &QQmlApplicationEngine::objectCreated,
        &app, [url](QObject *obj, const QUrl &objUrl) {{
            if (!obj && url == objUrl)
                QCoreApplication::exit(-1);
        }}, Qt::QueuedConnection);
    
    engine.load(url);

    return app.exec();
}}
"""

def generate_main_qml(project_name):
    return f"""import QtQuick
import QtQuick.Controls

ApplicationWindow {{
    visible: true
    width: 640
    height: 480
    title: qsTr("{project_name}")

    Column {{
        anchors.centerIn: parent
        spacing: 20

        Text {{
            text: qsTr("Hola, {project_name}!")
            font.pointSize: 24
            anchors.horizontalCenter: parent.horizontalCenter
        }}

        Button {{
            text: qsTr("Fes clic aquí")
            anchors.horizontalCenter: parent.horizontalCenter
            onClicked: {{
                console.log("Botó premut!")
            }}
        }}
    }}
}}
"""

if __name__ == "__main__":
    try:
        create_project()
    except KeyboardInterrupt:
        print("\n\n❌ Operació cancel·lada per l'usuari")
        sys.exit(1)
    except Exception as e:
        print(f"\n❌ Error: {e}")
        sys.exit(1)
