#include "mqttbasic.h"
#include <QDateTime>
#include <QDebug>

MqttBasic::MqttBasic(QObject *parent)
    : QObject(parent)
    , m_client(nullptr)
    , m_device(nullptr)
{
    vIniciaConnectaClientMqtt();
}

void MqttBasic::vIniciaConnectaClientMqtt(){
    // IMPORTANT: Només crear el client si no existeix
    if(m_client) {
        qDebug() << "Client MQTT ja existeix, no cal recrear-lo";
        return;
    }
    
    m_client = new QMqttClient(this);

    connect(m_client, &QMqttClient::stateChanged, this, &MqttBasic::updateLogStateChange);
    connect(m_client, &QMqttClient::disconnected, this, &MqttBasic::brokerDisconnected);

    connect(m_client, &QMqttClient::messageReceived, this, [this](const QByteArray &message, const QMqttTopicName &topic) {
        const QString content = QDateTime::currentDateTime().toString()
                    + QLatin1String(" Received Topic: ")
                    + topic.name()
                    + QLatin1String(" Message: ")
                    + message
                    + QLatin1Char('\n');
        emit vSignalMqttMsgRcv(topic.name(),message);
        qDebug() << content;
    });
}

void MqttBasic::vSetBroker(QString qsHost,int nPort,QString qsUsr,QString qsPwd){
    if(!m_client) {
        qDebug() << "Error: m_client és nullptr";
        return;
    }
    
    if (m_client->state() == QMqttClient::Disconnected){
        m_client->setHostname(qsHost);
        m_client->setPort(nPort);
        if(qsUsr!="" && qsPwd!=""){
            m_client->setUsername(qsUsr);
            m_client->setPassword(qsPwd);
        }
        m_client->connectToHost();
    }
}

void MqttBasic::vSetWsBroker(QString qsHost,QString qsUsr,QString qsPwd,QString qsMac){
    if(!m_client) {
        qDebug() << "Error: m_client és nullptr";
        return;
    }
    
    m_version = MQTT_VERSION;
    m_user = qsUsr;
    m_password = qsPwd;
    
    QUrl m_url = QUrl(qsHost);

    // Si ja existeix un device, tancar-lo primer
    if(m_device) {
        m_device->close();
        // NO fer delete perquè té parent!
        // El sistema de parents de Qt s'encarregarà d'això
        disconnect(m_device, nullptr, this, nullptr);
    }
    
    m_device = new WebSocketIODevice(this);
    m_device->setUrl(m_url);
    m_device->setProtocol("mqtt");

    connect(m_device, &WebSocketIODevice::socketConnected, this, [this]() {
        qDebug() << "WebSocket connected, initializing MQTT connection.";

        if(!m_client) {
            qDebug() << "Error: m_client és nullptr en callback";
            return;
        }

        m_client->setProtocolVersion(QMqttClient::MQTT_5_0);
        m_client->setTransport(m_device, QMqttClient::IODevice);
        m_client->setUsername(m_user);
        m_client->setPassword(m_password);

        m_client->connectToHost();
    });
    
    if (!m_device->open(QIODevice::ReadWrite)) {
        qDebug() << "Could not open socket device";
    }
}

void MqttBasic::vDisconnectBroker(){
    if(!m_client) {
        qDebug() << "Error: m_client és nullptr";
        return;
    }
    
    if (m_client->state() == QMqttClient::Connected){
        qDebug() << "Detecta que és connectat";
        m_client->disconnectFromHost();
    } else {
        qDebug() << "No detecta que és connectat";
    }
    
    if(m_device != nullptr){
        m_device->close();
        // NO fer delete perquè té parent!
        qDebug() << "Desconnectat";
        emit vSignalMqttDisconnected();
    }
}

MqttBasic::~MqttBasic()
{
    qDebug() << "Destructor MqttBasic";
    
    // Desconnectar primer
    if(m_client && m_client->state() == QMqttClient::Connected) {
        m_client->disconnectFromHost();
    }
    
    // Tancar device
    if(m_device) {
        m_device->close();
    }
    
    // Qt s'encarrega de destruir m_client i m_device perquè tenen parent (this)
}

void MqttBasic::updateLogStateChange(){
    if(!m_client) {
        qDebug() << "Error: m_client és nullptr en updateLogStateChange";
        return;
    }
    
    const QString content = QDateTime::currentDateTime().toString()
                    + QLatin1String(": State Change ")
                    + QString::number(m_client->state())
                    + QLatin1Char('\n');

    if(QString::number(m_client->state()) == "0"){
        qDebug() << "Arribo aquí 5";
        qDebug() << "Desconnectat";
        emit vSignalMqttDisconnected();
        qDebug() << "Arribo aquí 6";
    }
    if(QString::number(m_client->state()) == "1"){
        qDebug() << "Connectant";
        emit vSignalMqttConnecting();
    }
    if(QString::number(m_client->state()) == "2"){
        qDebug() << "Connectat";
        emit vSignalMqttConnected();
    }

    qDebug() << content;
}

void MqttBasic::brokerDisconnected(){
    qDebug() << "Arribo aquí 4";
    emit vSignalMqttDisconnected();
}

void MqttBasic::vTopicSubscription(QString qs){
    if(!m_client) {
        qDebug() << "Error: m_client és nullptr";
        return;
    }
    
    auto subscription = m_client->subscribe(qs);
    if (!subscription) {
        qDebug() << QLatin1String("Could not subscribe. Is there a valid connection?");
        return;
    }
}

void MqttBasic::vPublish(QString qsTopic,QString qsMsg){
    if(!m_client) {
        qDebug() << "Error: m_client és nullptr";
        return;
    }
    
    if (m_client->publish(qsTopic, qsMsg.toUtf8()) == -1)
        qDebug() << QLatin1String("Could not publish message");
}
